<?php

namespace App\Http\Controllers;

use App\Services\StripeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PaymentController extends Controller
{
    protected $stripeService;

    public function __construct(StripeService $stripeService)
    {
        $this->stripeService = $stripeService;
    }

    /**
     * Create a payment intent
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function createPaymentIntent(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:1',
            'currency' => 'required|string|size:3',
            'metadata' => 'nullable|array'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $result = $this->stripeService->createPaymentIntent(
            $request->amount,
            $request->currency,
            $request->metadata ?? []
        );

        if (!$result['success']) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create payment intent',
                'error' => $result['error']
            ], 500);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'client_secret' => $result['client_secret'],
                'payment_intent_id' => $result['payment_intent_id']
            ]
        ]);
    }

    /**
     * Retrieve a payment intent
     *
     * @param string $paymentIntentId
     * @return \Illuminate\Http\JsonResponse
     */
    public function retrievePaymentIntent($paymentIntentId)
    {
        $result = $this->stripeService->retrievePaymentIntent($paymentIntentId);

        if (!$result['success']) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve payment intent',
                'error' => $result['error']
            ], 500);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'payment_intent' => $result['payment_intent']
            ]
        ]);
    }

    /**
     * Confirm a payment intent
     *
     * @param string $paymentIntentId
     * @return \Illuminate\Http\JsonResponse
     */
    public function confirmPaymentIntent($paymentIntentId)
    {
        $result = $this->stripeService->confirmPaymentIntent($paymentIntentId);

        if (!$result['success']) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to confirm payment intent',
                'error' => $result['error']
            ], 500);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'payment_intent' => $result['payment_intent']
            ]
        ]);
    }

    /**
     * Cancel a payment intent
     *
     * @param string $paymentIntentId
     * @return \Illuminate\Http\JsonResponse
     */
    public function cancelPaymentIntent($paymentIntentId)
    {
        $result = $this->stripeService->cancelPaymentIntent($paymentIntentId);

        if (!$result['success']) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to cancel payment intent',
                'error' => $result['error']
            ], 500);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'payment_intent' => $result['payment_intent']
            ]
        ]);
    }

    /**
     * Get payment intent info (public endpoint)
     *
     * @param string $paymentIntentId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getPaymentIntentInfo($paymentIntentId)
    {
        $result = $this->stripeService->retrievePaymentIntent($paymentIntentId);

        if (!$result['success']) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve payment intent',
                'error' => $result['error']
            ], 500);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'amount' => $result['amount'],
                'currency' => $result['currency'],
                'status' => $result['status']
            ]
        ]);
    }
} 