<?php

namespace App\Http\Controllers;

use App\Models\Role;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class AdminUserController extends Controller
{
    /**
     * Display a paginated listing of users filtered by role type.
     *
     * This endpoint returns users based on their role. By default, it returns admin users,
     * but can be configured to return any role type via the 'role_type' query parameter.
     * Only superadmins can view superadmin users.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        // Get the requested role type (default to 'admin' if not specified)
        $roleType = $request->query('role_type', 'admin');
        
        // Validate the role type is one of the allowed values
        if (!in_array($roleType, ['admin', 'vendor', 'buyer', 'supervisor', 'superadmin'])) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid role type specified',
                'details' => 'Role type must be one of: admin, vendor, buyer, supervisor, superadmin'
            ], 400);
        }
        
        // Check permission for viewing superadmin users
        if ($roleType === 'superadmin' && !$request->user()->hasRole('superadmin')) {
            return response()->json([
                'success' => false,
                'message' => 'Access denied',
                'details' => 'You do not have permission to view superadmin users'
            ], 403);
        }
        
        // Get the role
        $role = Role::where('slug', $roleType)->first();
        
        if (!$role) {
            return response()->json([
                'success' => false,
                'message' => 'Role not found',
                'details' => "The {$roleType} role does not exist in the system"
            ], 404);
        }
        
        // Query users with the specified role and eager load profile
        $query = User::with('role');
        if ($roleType === 'vendor') {
            $query->with('vendorProfile');
        } elseif ($roleType === 'buyer') {
            $query->with('customerProfile');
        }
        $users = $query->where('role_id', $role->id)->paginate(15);

        // Add profile_id to each user
        $users->getCollection()->transform(function ($user) use ($roleType) {
            if ($roleType === 'vendor' && $user->vendorProfile) {
                $user->profile_id = $user->vendorProfile->id;
            } elseif ($roleType === 'buyer' && $user->customerProfile) {
                $user->profile_id = $user->customerProfile->id;
            } else {
                $user->profile_id = null;
            }
            return $user;
        });

        return response()->json([
            'success' => true,
            'data' => [
                'users' => $users,
                'role_type' => $roleType
            ]
        ]);
    }
    
    /**
     * Create a new user in the system.
     *
     * This endpoint allows creation of users with different roles based on the authenticated
     * user's permissions. Only superadmins can create other superadmin users.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        // Validate user input
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'username' => 'nullable|string|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'role_id' => 'required|exists:roles,id',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }
        
        // Check permission for creating users with specific roles
        $requestedRole = Role::find($request->role_id);
        if (!$requestedRole) {
            return response()->json([
                'success' => false,
                'message' => 'Role not found',
                'details' => 'The specified role_id does not exist in the system'
            ], 404);
        }
        
        $currentUser = $request->user();
        
        // Only superadmin can create other superadmins
        if ($requestedRole->slug === 'superadmin' && !$currentUser->hasRole('superadmin')) {
            return response()->json([
                'success' => false,
                'message' => 'Permission denied',
                'details' => 'Only superadmins can create superadmin users'
            ], 403);
        }
        
        // Create the user
        try {
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'username' => $request->username,
                'password' => Hash::make($request->password),
                'phone' => $request->phone,
                'address' => $request->address,
                'role_id' => $request->role_id,
                'is_active' => $request->has('is_active') ? $request->is_active : true,
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'User created successfully',
                'data' => [
                    'user' => $user->load('role')
                ]
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'User creation failed',
                'details' => 'An error occurred while creating the user',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Display the specified user.
     *
     * This endpoint retrieves a specific user by ID. It can optionally filter by role type.
     * Additional relationships (e.g., vendorProfile, buyerProfile) are loaded based on
     * the user's role. Only superadmins can view superadmin details.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Request $request, $id)
    {
        try {
            // Find the user by ID with their role relationship
            $user = User::with('role')->find($id);
            
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found',
                    'details' => "No user exists with ID: {$id}"
                ], 404);
            }
            
            // Get requested role type filter (optional)
            $roleType = $request->query('role_type');
            
            // If a specific role type is requested, verify the user has that role
            if ($roleType) {
                // Validate the role type is one of the allowed values
                if (!in_array($roleType, ['admin', 'vendor', 'buyer', 'supervisor', 'superadmin'])) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Invalid role type specified',
                        'details' => 'Role type must be one of: admin, vendor, buyer, supervisor, superadmin'
                    ], 400);
                }
                
                // Only superadmin can view superadmin details
                if ($roleType === 'superadmin' && !$request->user()->hasRole('superadmin')) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Access denied',
                        'details' => 'You do not have permission to view superadmin details'
                    ], 403);
                }
                
                // Check if user has the requested role
                if (!$user->hasRole($roleType)) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Role mismatch',
                        'details' => "The requested user is not a {$roleType}"
                    ], 400);
                }
            } else {
                // If no role type is specified but the user is a superadmin
                // Only superadmin can view superadmin details
                if ($user->hasRole('superadmin') && !$request->user()->hasRole('superadmin')) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Access denied',
                        'details' => 'You do not have permission to view superadmin details'
                    ], 403);
                }
            }
            
            // Load additional relationships based on user's role
            if ($user->hasRole('vendor')) {
                $user->load('vendorProfile');
            } elseif ($user->hasRole('buyer')) {
                $user->load('buyerProfile');
            }
            
            // Return the user data
            return response()->json([
                'success' => true,
                'data' => [
                    'user' => $user
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving user',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Update the specified user in the system.
     *
     * This endpoint allows updating user details. Special permission checks are enforced
     * for updating superadmin users or changing a user's role to superadmin. Password
     * updates are optional and only processed when explicitly provided.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            // Find the user by ID
            $user = User::find($id);
            
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found',
                    'details' => "No user exists with ID: {$id}"
                ], 404);
            }
            
            // Check permission for updating superadmin users
            if ($user->hasRole('superadmin') && !$request->user()->hasRole('superadmin')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Permission denied',
                    'details' => 'Only superadmins can update superadmin users'
                ], 403);
            }
            
            // Validate user input
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'email' => 'required|string|email|max:255|unique:users,email,' . $id,
                'username' => 'nullable|string|max:255|unique:users,username,' . $id,
                'password' => 'nullable|string|min:8|confirmed',
                'phone' => 'nullable|string|max:20',
                'address' => 'nullable|string',
                'role_id' => 'required|exists:roles,id',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Check permission for changing role to superadmin
            if ($request->has('role_id') && $request->role_id != $user->role_id) {
                $requestedRole = Role::find($request->role_id);
                if (!$requestedRole) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Role not found',
                        'details' => 'The specified role_id does not exist in the system'
                    ], 404);
                }
                
                if ($requestedRole->slug === 'superadmin' && !$request->user()->hasRole('superadmin')) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Permission denied',
                        'details' => 'Only superadmins can change a user to superadmin role'
                    ], 403);
                }
            }
            
            // Update the user
            $user->name = $request->name;
            $user->email = $request->email;
            $user->username = $request->username;
            $user->phone = $request->phone;
            $user->address = $request->address;
            $user->role_id = $request->role_id;
            
            if ($request->has('is_active')) {
                $user->is_active = $request->is_active;
            }
            
            if ($request->filled('password')) {
                $user->password = Hash::make($request->password);
            }
            
            $user->save();
            
            return response()->json([
                'success' => true,
                'message' => 'User updated successfully',
                'data' => [
                    'user' => $user->fresh()->load('role')
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'User update failed',
                'details' => 'An unexpected error occurred while updating the user',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Remove the specified user from the system.
     *
     * This endpoint deletes a user. Special checks prevent non-superadmins from
     * deleting superadmin users. Users cannot delete their own accounts.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            // Find the user by ID
            $user = User::find($id);
            
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found',
                    'details' => "No user exists with ID: {$id}"
                ], 404);
            }
            
            // Check permission for deleting superadmin users
            if ($user->hasRole('superadmin') && !request()->user()->hasRole('superadmin')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Permission denied',
                    'details' => 'Only superadmins can delete superadmin users'
                ], 403);
            }
            
            // Prevent self-deletion
            if ($user->id === request()->user()->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Operation not allowed',
                    'details' => 'You cannot delete your own account'
                ], 403);
            }
            
            $user->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'User deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'User deletion failed',
                'details' => 'An unexpected error occurred while deleting the user',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}