<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderHistory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class AdminOrderController extends Controller
{
    /**
     * Display a paginated list of all orders with filtering options.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            // Start building the query
            $query = Order::with(['user:id,name,email', 'items.product']);
            
            // Apply filters
            if ($request->has('status')) {
                $query->where('status', $request->status);
            }
            
            if ($request->has('payment_status')) {
                $query->where('payment_status', $request->payment_status);
            }
            
            if ($request->has('shipping_status')) {
                $query->where('shipping_status', $request->shipping_status);
            }
            
            // Filter by customer (user)
            if ($request->has('user_id')) {
                $query->where('user_id', $request->user_id);
            }
            
            // Filter by customer email (partial match)
            if ($request->has('email')) {
                $email = $request->email;
                $query->where('billing_email', 'like', "%{$email}%");
            }
            
            // Filter by date range
            if ($request->has('date_from')) {
                $query->where('created_at', '>=', $request->date_from);
            }
            
            if ($request->has('date_to')) {
                $query->where('created_at', '<=', $request->date_to . ' 23:59:59');
            }
            
            // Filter by order number (exact or partial match)
            if ($request->has('order_number')) {
                $orderNumber = $request->order_number;
                $query->where('order_number', 'like', "%{$orderNumber}%");
            }
            
            // Filter by total amount range
            if ($request->has('min_total')) {
                $query->where('total_amount', '>=', $request->min_total);
            }
            
            if ($request->has('max_total')) {
                $query->where('total_amount', '<=', $request->max_total);
            }
            
            // Apply sorting
            $sortField = $request->input('sort_by', 'created_at');
            $sortDirection = $request->input('sort_direction', 'desc');
            
            $validSortFields = [
                'id', 'order_number', 'created_at', 'status', 
                'payment_status', 'shipping_status', 'total_amount'
            ];
            
            if (in_array($sortField, $validSortFields)) {
                $query->orderBy($sortField, $sortDirection);
            } else {
                $query->orderBy('created_at', 'desc');
            }
            
            // Paginate the results
            $perPage = $request->input('per_page', 20);
            $orders = $query->paginate($perPage);
            
            // Get aggregated statistics for dashboard
            $stats = null;
            if ($request->boolean('include_stats', false)) {
                $stats = [
                    'total_orders' => Order::count(),
                    'pending_orders' => Order::where('status', 'pending')->count(),
                    'processing_orders' => Order::where('status', 'processing')->count(),
                    'completed_orders' => Order::where('status', 'completed')->count(),
                    'cancelled_orders' => Order::where('status', 'cancelled')->count(),
                    'total_revenue' => Order::where('payment_status', 'paid')->sum('total_amount'),
                    'recent_orders' => Order::where('created_at', '>=', now()->subDays(30))->count()
                ];
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'orders' => $orders,
                    'stats' => $stats
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error retrieving admin orders', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving orders',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }

    /**
     * Display the specified order with detailed information.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            // Find the order with all related data
            $order = Order::with([
                'user:id,name,email,phone',
                'items.product',
                'items.product.brand',
                'items.product.images',
                'history.user:id,name',
                'shipments',
                'fulfillmentOrders.warehouse',
                'fulfillmentOrders.items',
                'deliverySchedules'
            ])->findOrFail($id);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'order' => $order
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found',
                'error' => 'Not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error retrieving admin order details', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving order details',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }

    /**
     * Update the order status.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateStatus(Request $request, $id)
    {
        try {
            // Validate the request
            $validator = Validator::make($request->all(), [
                'status' => 'required|in:pending,processing,completed,cancelled,on_hold,refunded',
                'comment' => 'nullable|string|max:500',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }
            
            // Find the order
            $order = Order::findOrFail($id);
            
            // Check if the status is actually changing
            if ($order->status === $request->status) {
                return response()->json([
                    'success' => false,
                    'message' => 'Order status is already ' . $request->status,
                    'meta' => [
                        'code' => 400,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 400);
            }
            
            DB::beginTransaction();
            
            // Special handling for cancelled orders
            if ($request->status === 'cancelled' && $order->status !== 'cancelled') {
                // Return stock to inventory if order is being cancelled
                foreach ($order->items as $item) {
                    $product = $item->product;
                    if ($product) {
                        $product->quantity += $item->quantity;
                        
                        // Update stock status based on new quantity
                        if ($product->quantity > 0) {
                            if ($product->quantity <= 5) {
                                $product->stock_status = 'low_stock';
                            } else {
                                $product->stock_status = 'in_stock';
                            }
                        }
                        
                        $product->save();
                    }
                }
            }
            
            // Special handling for refunded orders
            if ($request->status === 'refunded') {
                // Only allow refunds for orders that have been paid
                if ($order->payment_status !== 'paid') {
                    DB::rollBack();
                    return response()->json([
                        'success' => false,
                        'message' => 'Cannot refund an order that has not been paid',
                        'meta' => [
                            'code' => 400,
                            'timestamp' => now()->toIso8601String(),
                        ],
                    ], 400);
                }
                
                // Update payment status to refunded
                $order->payment_status = 'refunded';
            }
            
            // Update the order status
            $oldStatus = $order->status;
            $order->status = $request->status;
            $order->save();
            
            // Create order history record
            $comment = $request->comment ?? 'Status changed from ' . $oldStatus . ' to ' . $request->status;
            $order->history()->create([
                'status' => $request->status,
                'comment' => $comment,
                'user_id' => $request->user()->id
            ]);
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Order status updated successfully',
                'data' => [
                    'order' => $order->fresh(['items.product', 'history'])
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found',
                'error' => 'Not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Error updating order status', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating order status',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }

    /**
     * Update the payment status of an order.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updatePaymentStatus(Request $request, $id)
    {
        try {
            // Validate the request
            $validator = Validator::make($request->all(), [
                'payment_status' => 'required|in:pending,processing,paid,failed,refunded,partially_refunded',
                'payment_transaction_id' => 'nullable|string|max:255',
                'refund_amount' => 'nullable|numeric|min:0',
                'refund_transaction_id' => 'nullable|string|max:255',
                'refund_reason' => 'nullable|string|max:500',
                'comment' => 'nullable|string|max:500',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }
            
            // Find the order
            $order = Order::findOrFail($id);
            
            // Check if the payment status is actually changing
            if ($order->payment_status === $request->payment_status && 
                !$request->has('payment_transaction_id') && 
                !$request->has('refund_amount')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment status is already ' . $request->payment_status,
                    'meta' => [
                        'code' => 400,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 400);
            }
            
            DB::beginTransaction();
            
            // Update payment status
            $oldPaymentStatus = $order->payment_status;
            $order->payment_status = $request->payment_status;
            
            // Update payment transaction ID if provided
            if ($request->has('payment_transaction_id')) {
                $order->payment_transaction_id = $request->payment_transaction_id;
            }
            
            // Handle refund information if applicable
            if (in_array($request->payment_status, ['refunded', 'partially_refunded'])) {
                if ($request->has('refund_amount')) {
                    $order->refund_amount = $request->refund_amount;
                }
                
                if ($request->has('refund_transaction_id')) {
                    $order->refund_transaction_id = $request->refund_transaction_id;
                }
                
                if ($request->has('refund_reason')) {
                    $order->refund_reason = $request->refund_reason;
                }
                
                // Validate refund amount for partially_refunded status
                if ($request->payment_status === 'partially_refunded') {
                    if (!$request->has('refund_amount') || $request->refund_amount <= 0) {
                        DB::rollBack();
                        return response()->json([
                            'success' => false,
                            'message' => 'Refund amount is required for partially refunded status',
                            'meta' => [
                                'code' => 400,
                                'timestamp' => now()->toIso8601String(),
                            ],
                        ], 400);
                    }
                    
                    if ($request->refund_amount >= $order->total_amount) {
                        DB::rollBack();
                        return response()->json([
                            'success' => false,
                            'message' => 'Refund amount must be less than total order amount for partially refunded status',
                            'meta' => [
                                'code' => 400,
                                'timestamp' => now()->toIso8601String(),
                            ],
                        ], 400);
                    }
                } else if ($request->payment_status === 'refunded') {
                    // For fully refunded, set the refund amount to total if not specified
                    if (!$request->has('refund_amount')) {
                        $order->refund_amount = $order->total_amount;
                    }
                }
            }
            
            $order->save();
            
            // Create order history record
            $comment = $request->comment ?? 'Payment status changed from ' . $oldPaymentStatus . ' to ' . $request->payment_status;
            $order->history()->create([
                'status' => 'payment_' . $request->payment_status,
                'comment' => $comment,
                'user_id' => $request->user()->id
            ]);
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Payment status updated successfully',
                'data' => [
                    'order' => $order->fresh(['items.product', 'history'])
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found',
                'error' => 'Not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Error updating payment status', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating payment status',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }

    /**
     * Update the shipping status of an order.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateShippingStatus(Request $request, $id)
    {
        try {
            // Validate the request
            $validator = Validator::make($request->all(), [
                'shipping_status' => 'required|in:pending,processing,packed,shipped,delivered,failed,returned',
                'tracking_number' => 'nullable|string|max:255',
                'tracking_url' => 'nullable|url|max:255',
                'carrier' => 'nullable|string|max:100',
                'shipping_date' => 'nullable|date',
                'estimated_delivery_date' => 'nullable|date',
                'actual_delivery_date' => 'nullable|date',
                'comment' => 'nullable|string|max:500',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                    'meta' => [
                        'code' => 422,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 422);
            }
            
            // Find the order
            $order = Order::findOrFail($id);
            
            // Check if the shipping status is actually changing
            if ($order->shipping_status === $request->shipping_status && 
                !$request->has('tracking_number') && 
                !$request->has('tracking_url')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Shipping status is already ' . $request->shipping_status,
                    'meta' => [
                        'code' => 400,
                        'timestamp' => now()->toIso8601String(),
                    ],
                ], 400);
            }
            
            DB::beginTransaction();
            
            // Update shipping status
            $oldShippingStatus = $order->shipping_status;
            $order->shipping_status = $request->shipping_status;
            
            // Create or update shipment if shipped status
            if ($request->shipping_status === 'shipped') {
                // Check if a shipment already exists
                $shipment = $order->shipments()->first();
                
                if (!$shipment) {
                    // Create a new shipment
                    $shipment = $order->shipments()->create([
                        'shipment_number' => 'SHP-' . strtoupper(substr(md5(uniqid()), 0, 8)),
                        'carrier' => $request->carrier,
                        'tracking_number' => $request->tracking_number,
                        'tracking_url' => $request->tracking_url,
                        'status' => 'shipped',
                        'shipped_date' => $request->shipping_date ?? now(),
                        'estimated_delivery_date' => $request->estimated_delivery_date,
                        'created_by' => $request->user()->id
                    ]);
                    
                    // Add items to shipment (include all order items)
                    foreach ($order->items as $item) {
                        $shipment->items()->create([
                            'order_item_id' => $item->id,
                            'quantity' => $item->quantity
                        ]);
                    }
                } else {
                    // Update existing shipment
                    $shipment->carrier = $request->carrier ?? $shipment->carrier;
                    $shipment->tracking_number = $request->tracking_number ?? $shipment->tracking_number;
                    $shipment->tracking_url = $request->tracking_url ?? $shipment->tracking_url;
                    $shipment->status = 'shipped';
                    $shipment->shipped_date = $request->shipping_date ?? $shipment->shipped_date ?? now();
                    $shipment->estimated_delivery_date = $request->estimated_delivery_date ?? $shipment->estimated_delivery_date;
                    $shipment->save();
                }
                
                // Add tracking record
                $shipment->tracking()->create([
                    'status' => 'shipped',
                    'location' => 'Warehouse',
                    'description' => 'Package has been shipped',
                    'timestamp' => now()
                ]);
            } else if ($request->shipping_status === 'delivered') {
                // Update shipment if exists
                $shipment = $order->shipments()->first();
                
                if ($shipment) {
                    $shipment->status = 'delivered';
                    $shipment->delivery_date = $request->actual_delivery_date ?? now();
                    $shipment->save();
                    
                    // Add tracking record
                    $shipment->tracking()->create([
                        'status' => 'delivered',
                        'location' => 'Destination',
                        'description' => 'Package has been delivered',
                        'timestamp' => now()
                    ]);
                    
                    // Add delivery attempt record
                    $shipment->deliveryAttempts()->create([
                        'attempt_date' => now(),
                        'status' => 'delivered',
                        'notes' => 'Package successfully delivered'
                    ]);
                }
                
                // If order status is still processing, update it to completed
                if ($order->status === 'processing') {
                    $order->status = 'completed';
                }
            }
            
            $order->save();
            
            // Create order history record
            $comment = $request->comment ?? 'Shipping status changed from ' . $oldShippingStatus . ' to ' . $request->shipping_status;
            $order->history()->create([
                'status' => 'shipping_' . $request->shipping_status,
                'comment' => $comment,
                'user_id' => $request->user()->id
            ]);
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Shipping status updated successfully',
                'data' => [
                    'order' => $order->fresh(['items.product', 'history', 'shipments.tracking'])
                ],
                'meta' => [
                    'code' => 200,
                    'timestamp' => now()->toIso8601String(),
                ],
            ]);
            
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found',
                'error' => 'Not found',
                'meta' => [
                    'code' => 404,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Error updating shipping status', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'order_id' => $id
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating shipping status',
                'error' => $e->getMessage(),
                'meta' => [
                    'code' => 500,
                    'timestamp' => now()->toIso8601String(),
                ],
            ], 500);
        }
    }
}