<?php

namespace App\Http\Controllers;

use App\Models\Brand;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class BrandController extends Controller
{
    /**
     * Display a listing of the brands.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $query = Brand::with('creator:id,name,email');
            
            // Filter by active status if requested
            if ($request->has('is_active')) {
                $query->where('is_active', $request->boolean('is_active'));
            }
            
            // Filter by featured status if requested
            if ($request->has('is_featured')) {
                $query->where('is_featured', $request->boolean('is_featured'));
            }
            
            // Search by name if provided
            if ($request->has('search')) {
                $query->where('name', 'like', '%' . $request->search . '%');
            }
            
            // Sorting by rank if requested
            if ($request->get('sort') === 'ranked') {
                $query->orderByRaw('CASE WHEN sort_order IS NULL OR sort_order = 0 THEN 1 ELSE 0 END, sort_order ASC, created_at DESC');
            } else {
                $query->orderBy('created_at', 'desc');
            }
            
            // Get paginated results
            $brands = $query->paginate(15);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'brands' => $brands
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving brands',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created brand in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            // Validate brand data
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255|unique:brands',
                'slug' => 'nullable|string|max:255|unique:brands',
                'logo' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'website' => 'nullable|url|max:255',
                'is_featured' => 'nullable|boolean',
                'is_active' => 'nullable|boolean',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Create slug from name if not provided
            $data = $request->all();
            if (empty($data['slug']) && !empty($data['name'])) {
                $data['slug'] = Str::slug($data['name']);
            }
            
            // Set default values
            $data['is_featured'] = $request->has('is_featured') ? $request->boolean('is_featured') : false;
            $data['is_active'] = $request->has('is_active') ? $request->boolean('is_active') : true;
            
            // Add the user ID of who created this brand
            $data['created_by'] = $request->user()->id;
            
            // Create the brand
            $brand = Brand::create($data);
            
            // Load the creator relationship for the response
            $brand->load('creator:id,name,email');
            
            return response()->json([
                'success' => true,
                'message' => 'Brand created successfully',
                'data' => [
                    'brand' => $brand
                ]
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Brand creation failed',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified brand.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $brand = Brand::findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => [
                    'brand' => [
                        'id' => $brand->id,
                        'name' => $brand->name,
                        'slug' => $brand->slug,
                        'description' => $brand->description,
                        'meta' => [
                            'title' => $brand->meta_title,
                            'description' => $brand->meta_description,
                            'keywords' => $brand->meta_keywords,
                        ],
                        'logo' => $brand->logo,
                        'is_active' => $brand->is_active,
                        'sort_order' => $brand->sort_order,
                    ]
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching brand', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error fetching brand',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified brand in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            // Find the brand
            $brand = Brand::findOrFail($id);
            
            // Validate brand data
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255|unique:brands,name,' . $id,
                'slug' => 'nullable|string|max:255|unique:brands,slug,' . $id,
                'logo' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'website' => 'nullable|url|max:255',
                'is_featured' => 'nullable|boolean',
                'is_active' => 'nullable|boolean',
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Update slug if name is changed and slug is not provided
            if ($request->has('name') && $request->name !== $brand->name && !$request->has('slug')) {
                $request->merge(['slug' => Str::slug($request->name)]);
            }
            
            // Update the brand
            $brand->update($request->all());
            
            return response()->json([
                'success' => true,
                'message' => 'Brand updated successfully',
                'data' => [
                    'brand' => $brand->fresh()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Brand update failed',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified brand from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            // Find the brand
            $brand = Brand::findOrFail($id);
            
            // Check if brand has associated products
            if ($brand->products()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete brand',
                    'details' => 'This brand has associated products. Please delete or reassign these products first.'
                ], 409);
            }
            
            // Delete the brand
            $brand->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Brand deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Brand deletion failed',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the rank (sort_order) of a brand (superadmin only).
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateRank(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'sort_order' => 'required|integer|min:1'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        // Only allow for superadmin
        if (!auth()->user()->hasRole(['superadmin'])) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $brand = Brand::findOrFail($id);
        $brand->sort_order = $request->sort_order;
        $brand->save();

        return response()->json([
            'success' => true,
            'message' => 'Brand rank updated successfully',
            'data' => [
                'brand' => $brand->fresh()
            ]
        ]);
    }

    /**
     * Bulk update brand ranks (superadmin only).
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function bulkUpdateRanks(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'ranks' => 'required|array|min:1',
            'ranks.*.id' => 'required|exists:brands,id',
            'ranks.*.sort_order' => 'required|integer|min:1',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        // Only allow for superadmin
        if (!auth()->user()->hasRole(['superadmin'])) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        foreach ($request->ranks as $rank) {
            Brand::where('id', $rank['id'])->update(['sort_order' => $rank['sort_order']]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Brand ranks updated successfully'
        ]);
    }

    /**
     * Update the meta JSON for a brand (Open Graph, Twitter, meta tags).
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateMeta(Request $request, $id)
    {
        $brand = Brand::findOrFail($id);

        $data = $request->validate([
            'meta' => 'required|array',
        ]);

        $brand->meta = $data['meta'];
        $brand->save();

        return response()->json([
            'success' => true,
            'meta' => $brand->meta,
        ]);
    }

    /**
     * Get the meta JSON for a brand (Open Graph, Twitter, meta tags).
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function getMeta($id)
    {
        $brand = Brand::findOrFail($id);

        return response()->json([
            'success' => true,
            'meta' => $brand->meta,
        ]);
    }
}