<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Product;
use App\Models\ProductPackPrice;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class CategoryController extends Controller
{
    /**
     * Display a listing of the categories.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $query = Category::query();
            
            // Filter by parent_id if provided
            if ($request->has('parent_id')) {
                $query->where('parent_id', $request->parent_id);
            }
            
            // Filter by featured status if provided
            if ($request->has('is_featured')) {
                $query->where('is_featured', $request->boolean('is_featured'));
            }
            
            // Filter by active status if provided
            if ($request->has('is_active')) {
                $query->where('is_active', $request->boolean('is_active'));
            }
            
            // Search by name if provided
            if ($request->has('search')) {
                $query->where('name', 'like', '%' . $request->search . '%');
            }
            
            // Include parent category if requested
            if ($request->boolean('with_parent', false)) {
                $query->with('parent');
            }
            
            // Include child categories if requested
            if ($request->boolean('with_children', false)) {
                $query->with('children');
            }
            
            // Include product count if requested
            if ($request->boolean('with_product_count', false)) {
                $query->withCount('products');
            }
            
            // Order by sort_order first (ranked categories), then by created_at for unranked categories
            $query->orderByRaw('CASE WHEN sort_order IS NULL OR sort_order = 0 THEN 1 ELSE 0 END, sort_order ASC, created_at DESC');
            
            // Get all categories or paginate if requested
            if ($request->boolean('paginate', true)) {
                $categories = $query->paginate($request->get('per_page', 15));
            } else {
                $categories = $query->get();
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Categories retrieved successfully',
                'data' => [
                    'categories' => $categories
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving categories: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving categories',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display a hierarchical tree of all categories.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function tree()
    {
        try {
            // Get all root categories (no parent)
            $categories = Category::whereNull('parent_id')
                ->with(['children' => function ($query) {
                    $query->orderBy('name', 'asc');
                }])
                ->orderBy('name', 'asc')
                ->get();
            
            return response()->json([
                'success' => true,
                'message' => 'Category tree retrieved successfully',
                'data' => [
                    'categories' => $categories
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving category tree: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving category tree',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created category in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'slug' => 'nullable|string|max:255|unique:categories,slug',
                'description' => 'nullable|string',
                'parent_id' => 'nullable|exists:categories,id',
                'image' => 'nullable|string|max:255',
                'is_featured' => 'nullable|boolean',
                'is_active' => 'nullable|boolean',
            ]);
            
            // Generate slug if not provided
            if (empty($validated['slug'])) {
                $slug = Str::slug($validated['name']);
                
                // Check if slug exists, append number if needed
                $count = 1;
                $originalSlug = $slug;
                while (Category::where('slug', $slug)->exists()) {
                    $slug = $originalSlug . '-' . $count++;
                }
                
                $validated['slug'] = $slug;
            }
            
            // Set default values for boolean fields
            $validated['is_featured'] = $validated['is_featured'] ?? false;
            $validated['is_active'] = $validated['is_active'] ?? true;
            
            // Create the category
            $category = Category::create($validated);
            
            return response()->json([
                'success' => true,
                'message' => 'Category created successfully',
                'data' => [
                    'category' => $category
                ]
            ], 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Error creating category: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error creating category',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified category.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $category = Category::with(['parent', 'children'])->findOrFail($id);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'category' => [
                        'id' => $category->id,
                        'name' => $category->name,
                        'slug' => $category->slug,
                        'description' => $category->description,
                        'meta' => [
                            'title' => $category->meta_title,
                            'description' => $category->meta_description,
                            'keywords' => $category->meta_keywords,
                        ],
                        'parent' => $category->parent ? [
                            'id' => $category->parent->id,
                            'name' => $category->parent->name,
                            'slug' => $category->parent->slug,
                        ] : null,
                        'children' => $category->children->map(function ($child) {
                            return [
                                'id' => $child->id,
                                'name' => $child->name,
                                'slug' => $child->slug,
                            ];
                        }),
                        'is_active' => $category->is_active,
                        'sort_order' => $category->sort_order,
                    ]
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching category', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error fetching category',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified category in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $category = Category::findOrFail($id);
            
            $validated = $request->validate([
                'name' => 'sometimes|required|string|max:255',
                'slug' => 'nullable|string|max:255|unique:categories,slug,' . $id,
                'description' => 'nullable|string',
                'parent_id' => [
                    'nullable',
                    'exists:categories,id',
                    function (
                        $attribute, $value, $fail
                    ) use ($id) {
                        if ($value == $id) {
                            $fail('A category cannot be its own parent.');
                        }
                        if ($value) {
                            $parent = Category::find($value);
                            while ($parent && $parent->parent_id) {
                                if ($parent->parent_id == $id) {
                                    $fail('This would create a circular reference in the category hierarchy.');
                                    break;
                                }
                                $parent = $parent->parent;
                            }
                        }
                    },
                ],
                'image' => 'nullable|string|max:255',
                'is_featured' => 'nullable|boolean',
                'is_active' => 'nullable|boolean',
                'sort_order' => 'nullable|integer|min:0',
            ]);
            
            // Generate slug if name changed and slug not provided
            if (isset($validated['name']) && !isset($validated['slug']) && $validated['name'] !== $category->name) {
                $slug = Str::slug($validated['name']);
                
                // Check if slug exists, append number if needed
                $count = 1;
                $originalSlug = $slug;
                while (Category::where('slug', $slug)->where('id', '!=', $id)->exists()) {
                    $slug = $originalSlug . '-' . $count++;
                }
                
                $validated['slug'] = $slug;
            }
            
            // Update the category
            $category->update($validated);
            
            return response()->json([
                'success' => true,
                'message' => 'Category updated successfully',
                'data' => [
                    'category' => $category->fresh()
                ]
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found',
                'details' => "No category exists with ID: {$id}"
            ], 404);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Error updating category: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error updating category',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified category from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            // Start a database transaction
            DB::beginTransaction();
            
            $category = Category::findOrFail($id);
            
            // Check if category has children
            $childrenCount = Category::where('parent_id', $id)->count();
            if ($childrenCount > 0) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete category with subcategories',
                    'details' => 'Please delete or reassign all subcategories first'
                ], 409);
            }
            
            // Handle products in this category
            // Update product_categories entries to remove this category
            DB::table('product_categories')->where('category_id', $id)->delete();
            
            // Delete the category
            $category->delete();
            
            // Commit the transaction
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Category deleted successfully'
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Category not found',
                'details' => "No category exists with ID: {$id}"
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting category: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error deleting category',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get featured categories.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function featured(Request $request)
    {
        try {
            $query = Category::where('is_featured', true)->where('is_active', true);
            
            // Include parent category if requested
            if ($request->boolean('with_parent', false)) {
                $query->with('parent');
            }
            
            // Include child categories if requested
            if ($request->boolean('with_children', false)) {
                $query->with('children');
            }
            
            // Limit the number of results if specified
            if ($request->has('limit')) {
                $categories = $query->limit($request->limit)->get();
            } else {
                $categories = $query->get();
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Featured categories retrieved successfully',
                'data' => [
                    'categories' => $categories
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving featured categories: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving featured categories',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update category status (active/featured).
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateStatus(Request $request, $id)
    {
        try {
            $category = Category::findOrFail($id);
            
            $validated = $request->validate([
                'is_active' => 'required_without:is_featured|boolean',
                'is_featured' => 'required_without:is_active|boolean',
            ]);
            
            // Update status
            if (isset($validated['is_active'])) {
                $category->is_active = $validated['is_active'];
            }
            
            if (isset($validated['is_featured'])) {
                $category->is_featured = $validated['is_featured'];
            }
            
            $category->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Category status updated successfully',
                'data' => [
                    'category' => $category
                ]
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found',
                'details' => "No category exists with ID: {$id}"
            ], 404);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Error updating category status: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error updating category status',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Perform bulk actions on categories (delete, activate, deactivate, etc).
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function bulkAction(Request $request)
    {
        try {
            $validated = $request->validate([
                'action' => 'required|in:delete,activate,deactivate,feature,unfeature',
                'category_ids' => 'required|array',
                'category_ids.*' => 'exists:categories,id',
            ]);
            
            $action = $validated['action'];
            $categoryIds = $validated['category_ids'];
            
            // Handle different actions
            switch ($action) {
                case 'delete':
                    // Check for subcategories
                    $hasChildren = Category::whereIn('parent_id', $categoryIds)->exists();
                    if ($hasChildren) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Cannot delete categories with subcategories',
                            'details' => 'Please delete or reassign all subcategories first'
                        ], 409);
                    }
                    
                    // Delete product_categories entries
                    DB::table('product_categories')->whereIn('category_id', $categoryIds)->delete();
                    
                    // Delete categories
                    Category::whereIn('id', $categoryIds)->delete();
                    $message = 'Categories deleted successfully';
                    break;
                    
                case 'activate':
                    Category::whereIn('id', $categoryIds)->update(['is_active' => true]);
                    $message = 'Categories activated successfully';
                    break;
                    
                case 'deactivate':
                    Category::whereIn('id', $categoryIds)->update(['is_active' => false]);
                    $message = 'Categories deactivated successfully';
                    break;
                    
                case 'feature':
                    Category::whereIn('id', $categoryIds)->update(['is_featured' => true]);
                    $message = 'Categories featured successfully';
                    break;
                    
                case 'unfeature':
                    Category::whereIn('id', $categoryIds)->update(['is_featured' => false]);
                    $message = 'Categories unfeatured successfully';
                    break;
                    
                default:
                    return response()->json([
                        'success' => false,
                        'message' => 'Invalid action',
                        'details' => 'The action provided is not supported'
                    ], 400);
            }
            
            return response()->json([
                'success' => true,
                'message' => $message
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Error performing bulk action on categories: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error performing bulk action',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all products belonging to a category.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function products(Request $request, $id)
    {
        try {
            // Find the category
            $category = Category::with('parent')->findOrFail($id);
            
            // Start with base query
            $query = Product::query()
                ->with(['brand', 'images', 'packPrice', 'bulkPrices', 'categories'])
                ->active()
                ->where('approval_status', 'approved');
                
            // Use the relationship to filter by category
            $query->whereHas('categories', function ($q) use ($id) {
                $q->where('categories.id', $id);
            });
            
            // Include products from subcategories if requested
            if ($request->boolean('include_subcategories', false)) {
                $subcategoryIds = $this->getAllSubcategoryIds($id);
                
                if (!empty($subcategoryIds)) {
                    $query->orWhereHas('categories', function ($q) use ($subcategoryIds) {
                        $q->whereIn('categories.id', $subcategoryIds);
                    });
                }
            }
            
            // Apply filters
            if ($request->has('brand_id')) {
                $query->byBrand($request->brand_id);
            }
            
            if ($request->has('vendor_id')) {
                $query->byVendor($request->vendor_id);
            }
            
            // Use the scoped price range filter
            if ($request->has('min_price') && $request->has('max_price')) {
                $query->byPriceRange($request->min_price, $request->max_price);
            } elseif ($request->has('min_price')) {
                $query->whereHas('packPrice', function ($q) use ($request) {
                    $q->where(function($subQ) use ($request) {
                        $subQ->whereNull('per_pack_special_price')
                            ->where('per_pack_price', '>=', $request->min_price);
                    })->orWhere(function($subQ) use ($request) {
                        $subQ->whereNotNull('per_pack_special_price')
                            ->where('per_pack_special_price', '>=', $request->min_price);
                    });
                });
            } elseif ($request->has('max_price')) {
                $query->whereHas('packPrice', function ($q) use ($request) {
                    $q->where(function($subQ) use ($request) {
                        $subQ->whereNull('per_pack_special_price')
                            ->where('per_pack_price', '<=', $request->max_price);
                    })->orWhere(function($subQ) use ($request) {
                        $subQ->whereNotNull('per_pack_special_price')
                            ->where('per_pack_special_price', '<=', $request->max_price);
                    });
                });
            }
            
            // Apply search filter using the scope
            if ($request->has('search')) {
                $query->search($request->search);
            }
            
            // Filter by stock status
            if ($request->has('stock_status')) {
                $stockStatus = $request->stock_status;
                if (in_array($stockStatus, ['in_stock', 'low_stock', 'out_of_stock'])) {
                    $query->where('stock_status', $stockStatus);
                }
            }
            
            // Filter by featured products
            if ($request->boolean('featured', false)) {
                $query->featured();
            }
            
            // Apply sorting
            $sortBy = $request->get('sort_by', 'created_at');
            $sortDirection = $request->get('sort_direction', 'desc');
            
            switch ($sortBy) {
                case 'name':
                    $query->orderBy('name', $sortDirection);
                    break;
                case 'price_low':
                    $query->join('product_pack_prices', 'products.id', '=', 'product_pack_prices.product_id')
                        ->select('products.*')
                        ->orderBy('product_pack_prices.per_pack_special_price', 'asc')
                        ->orderBy('product_pack_prices.per_pack_price', 'asc');
                    break;
                case 'price_high':
                    $query->join('product_pack_prices', 'products.id', '=', 'product_pack_prices.product_id')
                        ->select('products.*')
                        ->orderBy('product_pack_prices.per_pack_special_price', 'desc')
                        ->orderBy('product_pack_prices.per_pack_price', 'desc');
                    break;
                case 'discount':
                    $query->join('product_pack_prices', 'products.id', '=', 'product_pack_prices.product_id')
                        ->select('products.*')
                        ->whereNotNull('product_pack_prices.per_pack_special_price')
                        ->orderByRaw('(product_pack_prices.per_pack_price - product_pack_prices.per_pack_special_price) / product_pack_prices.per_pack_price DESC');
                    break;
                case 'newest':
                    $query->orderBy('created_at', 'desc');
                    break;
                case 'rating':
                    $query->orderBy('rating_average', 'desc');
                    break;
                default:
                    $query->orderBy('created_at', 'desc');
            }
            
            // Pagination
            $perPage = $request->get('per_page', 15);
            $products = $query->paginate($perPage);
            
            // Build breadcrumb for the category
            $breadcrumb = $category->getBreadcrumb()->map(function($item) {
                return [
                    'id' => $item->id,
                    'name' => $item->name,
                    'slug' => $item->slug
                ];
            });
            
            return response()->json([
                'success' => true,
                'message' => 'Products retrieved successfully',
                'data' => [
                    'category' => [
                        'id' => $category->id,
                        'name' => $category->name,
                        'slug' => $category->slug,
                        'description' => $category->description,
                        'image' => $category->image,
                        'breadcrumb' => $breadcrumb
                    ],
                    'products' => $products->items()
                ],
                'meta' => [
                    'total' => $products->total(),
                    'per_page' => $products->perPage(),
                    'current_page' => $products->currentPage(),
                    'last_page' => $products->lastPage(),
                    'from' => $products->firstItem(),
                    'to' => $products->lastItem(),
                    'path' => $products->path(),
                    'timestamp' => now()->toIso8601String()
                ]
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found',
                'details' => "No category exists with ID: {$id}"
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error retrieving category products: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving category products',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Helper method to get all subcategory IDs recursively.
     *
     * @param  int  $categoryId
     * @return array
     */
    private function getAllSubcategoryIds($categoryId)
    {
        $subcategoryIds = [];
        
        // Get immediate children
        $children = Category::where('parent_id', $categoryId)->get();
        
        foreach ($children as $child) {
            $subcategoryIds[] = $child->id;
            
            // Recursively get children of this child
            $childSubcategories = $this->getAllSubcategoryIds($child->id);
            if (!empty($childSubcategories)) {
                $subcategoryIds = array_merge($subcategoryIds, $childSubcategories);
            }
        }
        
        return $subcategoryIds;
    }

    /**
     * Bulk update sort_order for categories
     */
    public function bulkSortOrder(Request $request)
    {
        $validated = $request->validate([
            'orders' => 'required|array',
            'orders.*.id' => 'required|exists:categories,id',
            'orders.*.sort_order' => 'nullable|integer|min:0',
        ]);
        foreach ($validated['orders'] as $order) {
            Category::where('id', $order['id'])->update(['sort_order' => $order['sort_order']]);
        }
        return response()->json([
            'success' => true,
            'message' => 'Category sort orders updated successfully',
        ]);
    }

    /**
     * Bulk create categories and their subcategories.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function bulkCreate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'categories' => 'required|array|min:1',
            'categories.*.name' => 'required|string|max:255',
            'categories.*.description' => 'nullable|string',
            'categories.*.image' => 'nullable|string|max:255',
            'categories.*.is_featured' => 'nullable|boolean',
            'categories.*.is_active' => 'nullable|boolean',
            'categories.*.subcategories' => 'nullable|array',
            'categories.*.subcategories.*.name' => 'required_with:categories.*.subcategories|string|max:255',
            'categories.*.subcategories.*.description' => 'nullable|string',
            'categories.*.subcategories.*.image' => 'nullable|string|max:255',
            'categories.*.subcategories.*.is_featured' => 'nullable|boolean',
            'categories.*.subcategories.*.is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $created = [];
        foreach ($request->categories as $catData) {
            $cat = Category::create([
                'name' => $catData['name'],
                'description' => $catData['description'] ?? null,
                'image' => $catData['image'] ?? null,
                'is_featured' => $catData['is_featured'] ?? false,
                'is_active' => $catData['is_active'] ?? true,
            ]);
            $created[] = $cat;

            // Handle subcategories
            if (!empty($catData['subcategories'])) {
                foreach ($catData['subcategories'] as $subcatData) {
                    $subcat = Category::create([
                        'name' => $subcatData['name'],
                        'description' => $subcatData['description'] ?? null,
                        'image' => $subcatData['image'] ?? null,
                        'is_featured' => $subcatData['is_featured'] ?? false,
                        'is_active' => $subcatData['is_active'] ?? true,
                        'parent_id' => $cat->id,
                    ]);
                    $created[] = $subcat;
                }
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Categories created successfully',
            'data' => [
                'categories' => $created
            ]
        ], 201);
    }

    /**
     * Bulk update categories and their subcategories.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function bulkUpdate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'categories' => 'required|array|min:1',
            'categories.*.id' => 'required|exists:categories,id',
            'categories.*.name' => 'sometimes|string|max:255',
            'categories.*.description' => 'nullable|string',
            'categories.*.image' => 'nullable|string|max:255',
            'categories.*.is_featured' => 'nullable|boolean',
            'categories.*.is_active' => 'nullable|boolean',
            'categories.*.subcategories' => 'nullable|array',
            'categories.*.subcategories.*.id' => 'required_with:categories.*.subcategories|exists:categories,id',
            'categories.*.subcategories.*.name' => 'sometimes|string|max:255',
            'categories.*.subcategories.*.description' => 'nullable|string',
            'categories.*.subcategories.*.image' => 'nullable|string|max:255',
            'categories.*.subcategories.*.is_featured' => 'nullable|boolean',
            'categories.*.subcategories.*.is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $updated = [];
        foreach ($request->categories as $catData) {
            $cat = Category::findOrFail($catData['id']);
            $cat->update([
                'name' => $catData['name'] ?? $cat->name,
                'description' => $catData['description'] ?? $cat->description,
                'image' => $catData['image'] ?? $cat->image,
                'is_featured' => $catData['is_featured'] ?? $cat->is_featured,
                'is_active' => $catData['is_active'] ?? $cat->is_active,
            ]);
            $updated[] = $cat;

            // Handle subcategories
            if (!empty($catData['subcategories'])) {
                foreach ($catData['subcategories'] as $subcatData) {
                    $subcat = Category::findOrFail($subcatData['id']);
                    $subcat->update([
                        'name' => $subcatData['name'] ?? $subcat->name,
                        'description' => $subcatData['description'] ?? $subcat->description,
                        'image' => $subcatData['image'] ?? $subcat->image,
                        'is_featured' => $subcatData['is_featured'] ?? $subcat->is_featured,
                        'is_active' => $subcatData['is_active'] ?? $subcat->is_active,
                    ]);
                    $updated[] = $subcat;
                }
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Categories updated successfully',
            'data' => [
                'categories' => $updated
            ]
        ]);
    }

    /**
     * Update the meta JSON for a category (Open Graph, Twitter, meta tags).
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateMeta(Request $request, $id)
    {
        $category = Category::findOrFail($id);

        $data = $request->validate([
            'meta' => 'required|array',
        ]);

        $category->meta = $data['meta'];
        $category->save();

        return response()->json([
            'success' => true,
            'meta' => $category->meta,
        ]);
    }

    /**
     * Get the meta JSON for a category (Open Graph, Twitter, meta tags).
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function getMeta($id)
    {
        $category = Category::findOrFail($id);

        return response()->json([
            'success' => true,
            'meta' => $category->meta,
        ]);
    }
}