<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Brand;
use App\Models\Category;
use App\Models\ProductPackPrice;
use App\Models\ProductImage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class ExcelProductController extends Controller
{
    /**
     * Display a listing of Excel-imported products
     */
    public function index(Request $request)
    {
        $query = Product::with(['brand', 'categories', 'packPrice', 'images'])
            ->where('remarks', 'LIKE', '%Excel%');

        // Search functionality
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'LIKE', "%{$search}%")
                  ->orWhere('sku', 'LIKE', "%{$search}%")
                  ->orWhere('description', 'LIKE', "%{$search}%");
            });
        }

        // Filter by brand
        if ($request->has('brand_id') && $request->brand_id) {
            $query->where('brand_id', $request->brand_id);
        }

        // Filter by category
        if ($request->has('category_id') && $request->category_id) {
            $query->whereHas('categories', function($q) use ($request) {
                $q->where('category_id', $request->category_id);
            });
        }

        // Filter by stock status
        if ($request->has('stock_status') && $request->stock_status) {
            $query->where('stock_status', $request->stock_status);
        }

        // Sort options
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $products = $query->paginate(20);

        // Get brands and categories for filters
        $brands = Brand::where('is_active', 1)->get();
        $categories = Category::where('is_active', 1)->get();

        return response()->json([
            'success' => true,
            'data' => $products,
            'filters' => [
                'brands' => $brands,
                'categories' => $categories,
                'stock_statuses' => ['in_stock', 'out_of_stock', 'low_stock']
            ]
        ]);
    }

    /**
     * Show a specific Excel-imported product
     */
    public function show($id)
    {
        $product = Product::with(['brand', 'categories', 'packPrice', 'images', 'vendor'])
            ->where('remarks', 'LIKE', '%Excel%')
            ->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $product
        ]);
    }

    /**
     * Create a new Excel-imported product
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'sku' => 'required|string|max:100|unique:products,sku',
            'brand_id' => 'required|exists:brands,id',
            'description' => 'nullable|string',
            'quantity' => 'required|integer|min:0',
            'stock_status' => 'required|in:in_stock,out_of_stock,low_stock',
            'per_pack_price' => 'required|numeric|min:0',
            'number_of_products' => 'required|integer|min:1',
            'category_ids' => 'required|array|min:1',
            'category_ids.*' => 'exists:categories,id',
            'image_url' => 'nullable|url',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            // Create product
            $product = Product::create([
                'name' => $request->name,
                'title' => $request->name,
                'slug' => Str::slug($request->name),
                'sku' => $request->sku,
                'brand_id' => $request->brand_id,
                'vendor_id' => auth()->id(),
                'created_by' => auth()->id(),
                'description' => $request->description ?? $request->name,
                'quantity' => $request->quantity,
                'stock_status' => $request->stock_status,
                'meta_title' => $request->meta_title ?? $request->name,
                'meta_description' => $request->meta_description ?? $request->description,
                'is_approved' => 0,
                'approval_status' => 'pending',
                'is_active' => 1,
                'currency_code' => 'AUD',
                'remarks' => 'Created via Excel Product Controller',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Create pack price
            ProductPackPrice::create([
                'product_id' => $product->id,
                'number_of_products' => $request->number_of_products,
                'per_pack_price' => $request->per_pack_price,
                'per_pack_special_price' => null,
                'customer_margin' => 15.00,
                'partner_margin' => 10.00,
                'customer_margin_type' => 'percentage',
                'partner_margin_type' => 'percentage',
                'delivery_fee' => null,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Attach categories
            $product->categories()->attach($request->category_ids, [
                'sort_order' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Add image if provided
            if ($request->image_url) {
                ProductImage::create([
                    'product_id' => $product->id,
                    'image_url' => $request->image_url,
                    'alt_text' => $request->name,
                    'sort_order' => 1,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Excel product created successfully',
                'data' => $product->load(['brand', 'categories', 'packPrice', 'images'])
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to create product',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update an Excel-imported product
     */
    public function update(Request $request, $id)
    {
        $product = Product::where('remarks', 'LIKE', '%Excel%')
            ->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'sku' => 'sometimes|required|string|max:100|unique:products,sku,' . $id,
            'brand_id' => 'sometimes|required|exists:brands,id',
            'description' => 'nullable|string',
            'quantity' => 'sometimes|required|integer|min:0',
            'stock_status' => 'sometimes|required|in:in_stock,out_of_stock,low_stock',
            'per_pack_price' => 'sometimes|required|numeric|min:0',
            'number_of_products' => 'sometimes|required|integer|min:1',
            'category_ids' => 'sometimes|required|array|min:1',
            'category_ids.*' => 'exists:categories,id',
            'image_url' => 'nullable|url',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'is_active' => 'sometimes|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            // Update product
            $updateData = $request->only([
                'name', 'sku', 'brand_id', 'description', 'quantity', 
                'stock_status', 'meta_title', 'meta_description', 'is_active'
            ]);

            if (isset($updateData['name'])) {
                $updateData['title'] = $updateData['name'];
                $updateData['slug'] = Str::slug($updateData['name']);
            }

            $product->update($updateData);

            // Update pack price if provided
            if ($request->has('per_pack_price') || $request->has('number_of_products')) {
                $packPrice = $product->packPrice;
                if ($packPrice) {
                    $packPrice->update([
                        'per_pack_price' => $request->per_pack_price ?? $packPrice->per_pack_price,
                        'number_of_products' => $request->number_of_products ?? $packPrice->number_of_products,
                        'updated_at' => now(),
                    ]);
                }
            }

            // Update categories if provided
            if ($request->has('category_ids')) {
                $product->categories()->sync($request->category_ids);
            }

            // Update image if provided
            if ($request->image_url) {
                $existingImage = $product->images()->first();
                if ($existingImage) {
                    $existingImage->update([
                        'image_url' => $request->image_url,
                        'updated_at' => now(),
                    ]);
                } else {
                    ProductImage::create([
                        'product_id' => $product->id,
                        'image_url' => $request->image_url,
                        'alt_text' => $product->name,
                        'sort_order' => 1,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Excel product updated successfully',
                'data' => $product->load(['brand', 'categories', 'packPrice', 'images'])
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to update product',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete an Excel-imported product
     */
    public function destroy($id)
    {
        $product = Product::where('remarks', 'LIKE', '%Excel%')
            ->findOrFail($id);

        try {
            DB::beginTransaction();

            // Delete related records
            $product->packPrice()->delete();
            $product->categories()->detach();
            $product->images()->delete();
            
            // Delete the product
            $product->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Excel product deleted successfully'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete product',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk operations on Excel-imported products
     */
    public function bulkAction(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'action' => 'required|in:delete,activate,deactivate,update_stock',
            'product_ids' => 'required|array|min:1',
            'product_ids.*' => 'exists:products,id',
            'quantity' => 'required_if:action,update_stock|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            $products = Product::whereIn('id', $request->product_ids)
                ->where('remarks', 'LIKE', '%Excel%')
                ->get();

            $updatedCount = 0;

            foreach ($products as $product) {
                switch ($request->action) {
                    case 'delete':
                        $product->packPrice()->delete();
                        $product->categories()->detach();
                        $product->images()->delete();
                        $product->delete();
                        $updatedCount++;
                        break;

                    case 'activate':
                        $product->update(['is_active' => 1]);
                        $updatedCount++;
                        break;

                    case 'deactivate':
                        $product->update(['is_active' => 0]);
                        $updatedCount++;
                        break;

                    case 'update_stock':
                        $product->update([
                            'quantity' => $request->quantity,
                            'stock_status' => $request->quantity > 0 ? 'in_stock' : 'out_of_stock'
                        ]);
                        $updatedCount++;
                        break;
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Bulk action '{$request->action}' completed successfully on {$updatedCount} products"
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to perform bulk action',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get statistics for Excel-imported products
     */
    public function statistics()
    {
        $stats = DB::table('products')
            ->where('remarks', 'LIKE', '%Excel%')
            ->selectRaw('
                COUNT(*) as total_products,
                SUM(quantity) as total_stock,
                AVG(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) * 100 as active_percentage,
                COUNT(CASE WHEN stock_status = "in_stock" THEN 1 END) as in_stock_count,
                COUNT(CASE WHEN stock_status = "out_of_stock" THEN 1 END) as out_of_stock_count,
                COUNT(CASE WHEN stock_status = "low_stock" THEN 1 END) as low_stock_count
            ')
            ->first();

        $brandStats = DB::table('products')
            ->join('brands', 'products.brand_id', '=', 'brands.id')
            ->where('products.remarks', 'LIKE', '%Excel%')
            ->selectRaw('brands.name, COUNT(*) as product_count')
            ->groupBy('brands.id', 'brands.name')
            ->orderBy('product_count', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'overview' => $stats,
                'brands' => $brandStats
            ]
        ]);
    }
} 