<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Str;

class ImageController extends Controller
{
    /**
     * Upload and compress images
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function upload(Request $request)
    {
        try {
            // Validate request
            $validator = Validator::make($request->all(), [
                'images.*' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:10240', // Max 10MB per image
                'quality' => 'nullable|integer|min:1|max:100', // Compression quality
                'max_width' => 'nullable|integer|min:1', // Max width for resizing
                'max_height' => 'nullable|integer|min:1', // Max height for resizing
                'format' => 'nullable|in:jpeg,png,jpg,gif,webp', // Output format
                'webp_lossless' => 'nullable|boolean', // WebP lossless compression
                'webp_quality' => 'nullable|integer|min:0|max:100', // WebP quality (0-100)
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $quality = $request->input('quality', 80);
            $maxWidth = $request->input('max_width', 1920);
            $maxHeight = $request->input('max_height', 1080);
            $format = $request->input('format', 'jpeg');
            $webpLossless = $request->input('webp_lossless', false);
            $webpQuality = $request->input('webp_quality', 80);

            $uploadedImages = [];

            foreach ($request->file('images') as $image) {
                // Generate unique filename with appropriate extension
                $filename = Str::uuid() . '.' . $format;
                
                // Create image instance
                $img = Image::make($image);

                // Resize image if needed
                if ($img->width() > $maxWidth || $img->height() > $maxHeight) {
                    $img->resize($maxWidth, $maxHeight, function ($constraint) {
                        $constraint->aspectRatio();
                        $constraint->upsize();
                    });
                }

                // Compress and save image
                $path = 'uploads/images/' . $filename;
                
                if ($format === 'webp') {
                    // WebP specific encoding
                    $img->encode('webp', $webpLossless ? 100 : $webpQuality);
                } else {
                    // Regular format encoding
                    $img->encode($format, $quality);
                }

                Storage::disk('public')->put($path, $img->encode());

                $uploadedImages[] = [
                    'original_name' => $image->getClientOriginalName(),
                    'filename' => $filename,
                    'path' => 'storage/' . $path,
                    'size' => Storage::disk('public')->size($path),
                    'mime_type' => $image->getMimeType(),
                    'dimensions' => [
                        'width' => $img->width(),
                        'height' => $img->height()
                    ],
                    'format' => $format,
                    'compression' => [
                        'quality' => $format === 'webp' ? $webpQuality : $quality,
                        'lossless' => $format === 'webp' ? $webpLossless : null
                    ]
                ];
            }

            return response()->json([
                'success' => true,
                'message' => 'Images uploaded successfully',
                'data' => [
                    'images' => $uploadedImages
                ]
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error uploading images',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Serve images with CORS headers
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse|\Illuminate\Http\JsonResponse
     */
    public function show(Request $request)
    {
        try {
            // Validate request
            $validator = Validator::make($request->all(), [
                'urls' => 'required|array',
                'urls.*' => 'required|string|url'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $urls = $request->input('urls');
            $images = [];

            foreach ($urls as $url) {
                // Extract the path from the URL
                $path = parse_url($url, PHP_URL_PATH);
                
                // Remove the storage path prefix if it exists
                $path = str_replace('/storage/', '', $path);
                
                // Check if file exists in storage
                if (Storage::exists($path)) {
                    $images[] = [
                        'url' => $url,
                        'exists' => true,
                        'path' => $path,
                        'mime_type' => Storage::mimeType($path),
                        'size' => Storage::size($path)
                    ];
                } else {
                    $images[] = [
                        'url' => $url,
                        'exists' => false,
                        'error' => 'Image not found'
                    ];
                }
            }

            // If only one image is requested, serve it directly
            if (count($urls) === 1 && $images[0]['exists']) {
                $path = $images[0]['path'];
                $mimeType = $images[0]['mime_type'];
                
                // Set CORS headers
                $headers = [
                    'Access-Control-Allow-Origin' => '*',
                    'Access-Control-Allow-Methods' => 'GET, OPTIONS',
                    'Access-Control-Allow-Headers' => 'Content-Type, Authorization',
                    'Content-Type' => $mimeType,
                    'Cache-Control' => 'public, max-age=31536000'
                ];
                
                return response()->file(Storage::path($path), $headers);
            }

            // For multiple images, return their information
            return response()->json([
                'success' => true,
                'data' => [
                    'images' => $images
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving images',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Handle OPTIONS request for CORS preflight
     *
     * @return \Illuminate\Http\Response
     */
    public function handleCors()
    {
        return response()->json(null, 200)
            ->header('Access-Control-Allow-Origin', '*')
            ->header('Access-Control-Allow-Methods', 'GET, OPTIONS')
            ->header('Access-Control-Allow-Headers', 'Content-Type, Authorization');
    }
} 