<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\User;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class SuperAdminDashboardController extends Controller
{
    /**
     * Get dashboard metrics for superadmin.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function metrics(Request $request)
    {
        // Sales data
        $now = Carbon::now();
        $oneWeekAgo = $now->copy()->subWeek();
        $oneMonthAgo = $now->copy()->subMonth();
        $sixMonthsAgo = $now->copy()->subMonths(6);

        // Orders
        $orders = Order::select('total_amount', 'created_at')->get();
        $ordersLastWeek = $orders->where('created_at', '>=', $oneWeekAgo);
        $ordersLastMonth = $orders->where('created_at', '>=', $oneMonthAgo);
        $ordersLast6Months = $orders->where('created_at', '>=', $sixMonthsAgo);

        // Sales sums
        $salesLastWeek = $ordersLastWeek->sum('total_amount');
        $salesLastMonth = $ordersLastMonth->sum('total_amount');
        $salesLast6Months = $ordersLast6Months->sum('total_amount');
        $totalOrderValue = $orders->sum('total_amount');
        $totalOrders = $orders->count();

        // Vendors
        $totalVendors = User::whereHas('role', function($q) {
            $q->where('slug', 'vendor');
        })->count();

        // Products
        $totalProducts = Product::count();
        $totalApprovedProducts = Product::where('approval_status', 'approved')->count();
        $totalActiveProducts = Product::where('is_active', true)->count();

        // Customers
        $totalCustomers = User::whereHas('role', function($q) {
            $q->where('slug', 'buyer');
        })->count();

        // Top 5 selling products (by order items)
        $topProducts = DB::table('order_items')
            ->select('product_id', DB::raw('SUM(quantity) as total_sold'))
            ->groupBy('product_id')
            ->orderByDesc('total_sold')
            ->limit(5)
            ->get();

        // Top 5 vendors by sales
        $topVendors = DB::table('orders')
            ->join('order_items', 'orders.id', '=', 'order_items.order_id')
            ->join('products', 'order_items.product_id', '=', 'products.id')
            ->select('products.vendor_id', DB::raw('SUM(order_items.total_amount) as vendor_sales'))
            ->groupBy('products.vendor_id')
            ->orderByDesc('vendor_sales')
            ->limit(5)
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'sales' => [
                    'last_week' => $salesLastWeek,
                    'last_month' => $salesLastMonth,
                    'last_6_months' => $salesLast6Months,
                    'total_order_value' => $totalOrderValue,
                    'total_orders' => $totalOrders,
                ],
                'vendors' => [
                    'total' => $totalVendors,
                    'top_vendors' => $topVendors,
                ],
                'products' => [
                    'total' => $totalProducts,
                    'approved' => $totalApprovedProducts,
                    'active' => $totalActiveProducts,
                    'top_selling' => $topProducts,
                ],
                'customers' => [
                    'total' => $totalCustomers,
                ],
            ]
        ]);
    }
} 