<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class VendorAddressController extends Controller
{
    /**
     * Get the vendor's address.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $vendor = $request->user();
            
            if (!$vendor->isVendor()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized',
                    'details' => 'Only vendors can access this endpoint'
                ], 403);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'address' => $vendor->address,
                    'phone' => $vendor->phone
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving address',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the vendor's address.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $vendor = $request->user();

            if (!$vendor->isVendor()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized',
                    'details' => 'Only vendors can access this endpoint'
                ], 403);
            }

            // Validate the request
            $validator = Validator::make($request->all(), [
                'address_line1' => 'required|string|max:255',
                'address_line2' => 'nullable|string|max:255',
                'city' => 'required|string|max:100',
                'state' => 'required|string|max:100',
                'postal_code' => 'required|string|max:20',
                'country' => 'required|string|max:100',
                'phone' => 'nullable|string|max:20'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Format the full address
            $address = $request->address_line1;
            if ($request->address_line2) {
                $address .= ', ' . $request->address_line2;
            }
            $address .= ', ' . $request->city;
            $address .= ', ' . $request->state;
            $address .= ' ' . $request->postal_code;
            $address .= ', ' . $request->country;

            // Update user's address and phone
            $vendor->update([
                'address' => $address,
                'phone' => $request->phone
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Address updated successfully',
                'data' => [
                    'address' => $vendor->address,
                    'phone' => $vendor->phone
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating address',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get the vendor's address.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Request $request)
    {
        return $this->index($request);
    }

    /**
     * Update the vendor's address.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request)
    {
        return $this->store($request);
    }

    /**
     * Delete the vendor's address.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(Request $request)
    {
        try {
            $vendor = $request->user();

            if (!$vendor->isVendor()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized',
                    'details' => 'Only vendors can access this endpoint'
                ], 403);
            }

            // Clear address and phone
            $vendor->update([
                'address' => null,
                'phone' => null
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Address deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting address',
                'details' => 'An unexpected error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Set the vendor's address as default.
     * This method is kept for API compatibility but doesn't do anything
     * since vendors only have one address.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function setDefault(Request $request)
    {
        return response()->json([
            'success' => true,
            'message' => 'Address is already the default address'
        ]);
    }
} 