<?php

namespace App\Http\Controllers;

use App\Models\Wishlist;
use App\Models\WishlistItem;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class WishlistController extends Controller
{
    /**
     * Display a listing of the customer's wishlists.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $userId = $request->user()->id;
            
            $wishlists = Wishlist::with('items.product')
                ->where('user_id', $userId)
                ->get();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'wishlists' => $wishlists
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving wishlists',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Store a newly created wishlist.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $userId = $request->user()->id;
            
            // Validate the request
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'is_public' => 'nullable|boolean',
                'products' => 'nullable|array',
                'products.*.product_id' => 'required|exists:products,id',
                'products.*.quantity' => 'nullable|integer|min:1',
                'products.*.notes' => 'nullable|string'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Create the wishlist
            $wishlist = Wishlist::create([
                'user_id' => $userId,
                'name' => $request->name,
                'is_public' => $request->boolean('is_public', false)
            ]);
            
            // Add products if provided
            if ($request->has('products') && is_array($request->products)) {
                foreach ($request->products as $productData) {
                    WishlistItem::create([
                        'wishlist_id' => $wishlist->id,
                        'product_id' => $productData['product_id'],
                        'quantity' => $productData['quantity'] ?? 1,
                        'notes' => $productData['notes'] ?? null
                    ]);
                }
            }
            
            // Load the wishlist with its items
            $wishlist->load('items.product');
            
            return response()->json([
                'success' => true,
                'message' => 'Wishlist created successfully',
                'data' => [
                    'wishlist' => $wishlist
                ]
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating wishlist',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Display the specified wishlist.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $userId = auth()->id();
            
            $wishlist = Wishlist::with('items.product')
                ->where('id', $id)
                ->where(function($query) use ($userId) {
                    $query->where('user_id', $userId)
                          ->orWhere('is_public', true);
                })
                ->firstOrFail();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'wishlist' => $wishlist,
                    'is_owner' => $wishlist->user_id === $userId
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Wishlist not found',
                'error' => $e->getMessage()
            ], 404);
        }
    }
    
    /**
     * Update the specified wishlist.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $userId = $request->user()->id;
            
            // Find the wishlist
            $wishlist = Wishlist::where('id', $id)
                ->where('user_id', $userId)
                ->firstOrFail();
            
            // Validate the request
            $validator = Validator::make($request->all(), [
                'name' => 'nullable|string|max:255',
                'is_public' => 'nullable|boolean'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Update the wishlist
            if ($request->has('name')) {
                $wishlist->name = $request->name;
            }
            
            if ($request->has('is_public')) {
                $wishlist->is_public = $request->boolean('is_public');
            }
            
            $wishlist->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Wishlist updated successfully',
                'data' => [
                    'wishlist' => $wishlist->fresh(['items.product'])
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating wishlist',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Remove the specified wishlist.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $userId = auth()->id();
            
            $wishlist = Wishlist::where('id', $id)
                ->where('user_id', $userId)
                ->firstOrFail();
            
            // Delete the wishlist (items will be cascade deleted)
            $wishlist->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Wishlist deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting wishlist',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Add a product to the wishlist.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function addProduct(Request $request, $id)
    {
        try {
            $userId = $request->user()->id;
            
            // Find the wishlist
            $wishlist = Wishlist::where('id', $id)
                ->where('user_id', $userId)
                ->firstOrFail();
            
            // Validate the request
            $validator = Validator::make($request->all(), [
                'product_id' => 'required|exists:products,id',
                'quantity' => 'nullable|integer|min:1',
                'notes' => 'nullable|string'
            ]);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            // Check if product already exists in wishlist
            $existingItem = WishlistItem::where('wishlist_id', $id)
                ->where('product_id', $request->product_id)
                ->first();
            
            if ($existingItem) {
                // Update quantity if exists
                $existingItem->quantity = $request->quantity ?? $existingItem->quantity;
                $existingItem->notes = $request->notes ?? $existingItem->notes;
                $existingItem->save();
                
                $item = $existingItem;
                $message = 'Product updated in wishlist';
            } else {
                // Add new product to wishlist
                $item = WishlistItem::create([
                    'wishlist_id' => $wishlist->id,
                    'product_id' => $request->product_id,
                    'quantity' => $request->quantity ?? 1,
                    'notes' => $request->notes ?? null
                ]);
                
                $message = 'Product added to wishlist';
            }
            
            // Load the item with product
            $item->load('product');
            
            return response()->json([
                'success' => true,
                'message' => $message,
                'data' => [
                    'item' => $item
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error adding product to wishlist',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Remove a product from the wishlist.
     *
     * @param  int  $wishlistId
     * @param  int  $productId
     * @return \Illuminate\Http\JsonResponse
     */
    public function removeProduct($wishlistId, $productId)
    {
        try {
            $userId = auth()->id();
            
            // Find the wishlist
            $wishlist = Wishlist::where('id', $wishlistId)
                ->where('user_id', $userId)
                ->firstOrFail();
            
            // Find and delete the item
            $deleted = WishlistItem::where('wishlist_id', $wishlistId)
                ->where('product_id', $productId)
                ->delete();
            
            if (!$deleted) {
                return response()->json([
                    'success' => false,
                    'message' => 'Product not found in wishlist'
                ], 404);
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Product removed from wishlist'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error removing product from wishlist',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}