<?php

namespace App\Http\Controllers;

use App\Models\VendorMembership;
use Illuminate\Http\Request;

class MembershipController extends Controller
{
    // List all memberships
    public function index()
    {
        return response()->json([
            'success' => true,
            'data' => VendorMembership::all()
        ]);
    }

    // Create a new membership
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100|unique:vendor_memberships,name',
            'normal_discount' => 'required|numeric|min:0|max:100',
            'extra_discount' => 'required|numeric|min:0|max:100',
        ]);
        $membership = VendorMembership::create($validated);
        return response()->json([
            'success' => true,
            'data' => $membership
        ], 201);
    }

    // Get a specific membership
    public function show($id)
    {
        $membership = VendorMembership::findOrFail($id);
        return response()->json([
            'success' => true,
            'data' => $membership
        ]);
    }

    // Update an existing membership
    public function update(Request $request, $id)
    {
        $membership = VendorMembership::findOrFail($id);
        $validated = $request->validate([
            'name' => 'required|string|max:100|unique:vendor_memberships,name,' . $id,
            'normal_discount' => 'required|numeric|min:0|max:100',
            'extra_discount' => 'required|numeric|min:0|max:100',
        ]);
        $membership->update($validated);
        return response()->json([
            'success' => true,
            'data' => $membership
        ]);
    }

    // Delete a membership
    public function destroy($id)
    {
        $membership = VendorMembership::findOrFail($id);
        $membership->delete();
        return response()->json([
            'success' => true,
            'message' => 'Membership deleted.'
        ]);
    }

    // Get all members (vendors) under a specific membership
    public function members($id)
    {
        $membership = \App\Models\VendorMembership::with('vendorProfiles.user')->findOrFail($id);
        return response()->json([
            'success' => true,
            'data' => $membership->vendorProfiles
        ]);
    }

    // Assign multiple vendors to a membership package
    public function bulkAssign(Request $request)
    {
        $request->validate([
            'membership_id' => 'required|exists:vendor_memberships,id',
            'vendor_profile_ids' => 'required|array|min:1',
            'vendor_profile_ids.*' => 'required|exists:vendor_profiles,id',
        ]);

        $membershipId = $request->membership_id;
        $vendorProfileIds = $request->vendor_profile_ids;

        $updatedCount = \App\Models\VendorProfile::whereIn('id', $vendorProfileIds)
            ->update(['membership_id' => $membershipId]);

        $updatedVendors = \App\Models\VendorProfile::whereIn('id', $vendorProfileIds)
            ->with('user')
            ->get();

        return response()->json([
            'success' => true,
            'message' => $updatedCount . ' vendor(s) assigned to membership package.',
            'updated_count' => $updatedCount,
            'data' => $updatedVendors
        ]);
    }

    // Remove a vendor's membership
    public function removeVendorMembership($vendorProfileId)
    {
        $vendorProfile = \App\Models\VendorProfile::findOrFail($vendorProfileId);
        
        $vendorProfile->membership_id = null;
        $vendorProfile->save();

        return response()->json([
            'success' => true,
            'message' => 'Vendor membership removed successfully.',
            'data' => $vendorProfile->load('user')
        ]);
    }
} 