<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CustomerPaymentMethod extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'payment_type',
        'is_default',
        'provider',
        'account_name',
        'account_number_last4',
        'expiry_month',
        'expiry_year',
        'token',
        'billing_address_id',
        'is_verified'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_default' => 'boolean',
        'is_verified' => 'boolean',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'token',
    ];

    /**
     * Get the user that owns the payment method.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the customer profile associated with this payment method.
     */
    public function customerProfile()
    {
        return $this->belongsTo(CustomerProfile::class, 'user_id', 'user_id');
    }

    /**
     * Get the billing address for this payment method.
     */
    public function billingAddress()
    {
        return $this->belongsTo(CustomerAddress::class, 'billing_address_id');
    }

    /**
     * Scope a query to only include default payment methods.
     */
    public function scopeDefault($query)
    {
        return $query->where('is_default', true);
    }

    /**
     * Scope a query to filter by payment type.
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('payment_type', $type);
    }

    /**
     * Get masked account number.
     */
    public function getMaskedAccountNumberAttribute()
    {
        if ($this->account_number_last4) {
            return '••••••••' . $this->account_number_last4;
        }
        
        return null;
    }

    /**
     * Get expiry date formatted as MM/YY.
     */
    public function getExpiryDateAttribute()
    {
        if ($this->expiry_month && $this->expiry_year) {
            return $this->expiry_month . '/' . substr($this->expiry_year, -2);
        }
        
        return null;
    }
}