<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ProductBulkPrice extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'product_id',
        'number_of_packs',
        'per_pack_price',
        'per_pack_special_price',
        'customer_margin',
        'partner_margin',
        'customer_margin_type',
        'partner_margin_type',
        'sort_order',
        'delivery_fee',
        'tax_rate',
        
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'number_of_packs' => 'integer',
        'per_pack_price' => 'decimal:2',
        'per_pack_special_price' => 'decimal:2',
        'customer_margin' => 'decimal:2',
        'partner_margin' => 'decimal:2',
        'customer_margin_type' => 'string',
        'partner_margin_type' => 'string',
        'sort_order' => 'integer',
        'delivery_fee' => 'decimal:2',
        'tax_rate' => 'decimal:2',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        'total_price',
        'total_special_price',
        'discount_percentage',
        'customer_margin_amount',
        'partner_margin_amount',
    ];

    /**
     * Boot function from Laravel.
     */
    protected static function boot()
    {
        parent::boot();
        
        // When creating a bulk price, set default margin types if not provided
        static::creating(function ($bulkPrice) {
            if (empty($bulkPrice->customer_margin_type)) {
                $bulkPrice->customer_margin_type = 'percentage';
            }
            
            if (empty($bulkPrice->partner_margin_type)) {
                $bulkPrice->partner_margin_type = 'percentage';
            }
            
            // Set default margins from product if not provided
            if ($bulkPrice->product_id && (is_null($bulkPrice->customer_margin) || is_null($bulkPrice->partner_margin))) {
                $product = Product::find($bulkPrice->product_id);
                if ($product) {
                    if (is_null($bulkPrice->customer_margin)) {
                        $customerMargin = $product->getDefaultCustomerMargin();
                        $bulkPrice->customer_margin = $customerMargin['value'];
                        $bulkPrice->customer_margin_type = $customerMargin['type'];
                    }
                    
                    if (is_null($bulkPrice->partner_margin)) {
                        $partnerMargin = $product->getDefaultPartnerMargin();
                        $bulkPrice->partner_margin = $partnerMargin['value'];
                        $bulkPrice->partner_margin_type = $partnerMargin['type'];
                    }
                }
            }
        });
    }

    /**
     * Get the product that owns the bulk price.
     */
    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * Get the offers for this bulk price.
     */
    public function offers()
    {
        return $this->hasMany(ProductOffer::class, 'price_id')
            ->where('price_type', 'bulk');
    }

    /**
     * Get the active offers for this bulk price.
     */
    public function activeOffers()
    {
        return $this->offers()
            ->where('is_active', true)
            ->where(function ($query) {
                $query->whereNull('start_date')
                    ->orWhere('start_date', '<=', now());
            })
            ->where(function ($query) {
                $query->whereNull('end_date')
                    ->orWhere('end_date', '>=', now());
            });
    }

    /**
     * Get the best active offer for this bulk price.
     */
    public function bestOffer()
    {
        return $this->activeOffers()
            ->orderBy('price')
            ->first();
    }

    /**
     * Get the final per-pack price after applying the best offer.
     */
    public function getFinalPrice()
    {
        $bestOffer = $this->bestOffer();
        if ($bestOffer) {
            return $bestOffer->price;
        }
        
        return $this->per_pack_special_price ?? $this->per_pack_price;
    }

    /**
     * Get the discount percentage.
     */
    public function getDiscountPercentage()
    {
        if (!$this->per_pack_special_price) {
            return 0;
        }
        
        if ($this->per_pack_price <= 0) {
            return 0;
        }
        
        return round((($this->per_pack_price - $this->per_pack_special_price) / $this->per_pack_price) * 100, 2);
    }

    /**
     * Get the discount percentage attribute.
     */
    public function getDiscountPercentageAttribute()
    {
        return $this->getDiscountPercentage();
    }

    /**
     * Get the total price (regular price × quantity).
     */
    public function getTotalPriceAttribute()
    {
        return $this->per_pack_price * $this->number_of_packs;
    }

    /**
     * Get the total special price (special price × quantity).
     */
    public function getTotalSpecialPriceAttribute()
    {
        return round($this->per_pack_special_price * $this->number_of_packs, 2);
    }

    /**
     * Get the tax rate for this bulk price (inherits from product if not set)
     */
    public function getTaxRate()
    {
        // Use bulk price specific tax rate if set, otherwise inherit from product
        if (!is_null($this->tax_rate)) {
            return $this->tax_rate;
        }
        
        // Fall back to product tax rate
        return $this->product->getTaxRate();
    }

    /**
     * Calculate customer margin amount based on margin type.
     */
    public function getCustomerMarginAmountAttribute()
    {
        $basePrice = $this->per_pack_special_price ?? $this->per_pack_price;
        
        if ($this->customer_margin_type === 'percentage') {
            // Calculate percentage margin
            return round(($basePrice * $this->customer_margin) / 100, 2);
        } else {
            // Return fixed margin
            return $this->customer_margin;
        }
    }
    
    /**
     * Calculate partner margin amount based on margin type.
     */
    public function getPartnerMarginAmountAttribute()
    {
        $basePrice = $this->per_pack_special_price ?? $this->per_pack_price;
        
        if ($this->partner_margin_type === 'percentage') {
            // Calculate percentage margin
            return round(($basePrice * $this->partner_margin) / 100, 2);
        } else {
            // Return fixed margin
            return $this->partner_margin;
        }
    }

    /**
     * Format the price with currency symbol.
     */
    public function formatPrice($price)
    {
        if ($this->product) {
            return $this->product->formatPrice($price);
        }
        
        // Fallback if product is not loaded
        return '$' . number_format($price, 2);
    }

    /**
     * Scope a query to order by number of packs.
     */
    public function scopeOrderByPackQuantity($query, $direction = 'asc')
    {
        return $query->orderBy('number_of_packs', $direction);
    }

    /**
     * Scope a query to order by price per pack.
     */
    public function scopeOrderByPrice($query, $direction = 'asc')
    {
        return $query->orderBy('per_pack_price', $direction);
    }

    /**
     * Scope a query to order by sort order.
     */
    public function scopeOrderBySortOrder($query)
    {
        return $query->orderBy('sort_order');
    }

    /**
     * Calculate savings compared to buying at the pack price.
     * 
     * @param ProductPackPrice $packPrice
     * @return array
     */
    public function calculateSavings(ProductPackPrice $packPrice)
    {
        if (!$packPrice || $packPrice->per_pack_price <= 0) {
            return [
                'amount' => 0,
                'percentage' => 0
            ];
        }

        $packRegularPrice = $packPrice->per_pack_price;
        $bulkRegularPrice = $this->per_pack_price;
        
        $amountSaved = $packRegularPrice - $bulkRegularPrice;
        $percentageSaved = round(($amountSaved / $packRegularPrice) * 100, 2);
        
        return [
            'amount' => $amountSaved,
            'percentage' => $percentageSaved
        ];
    }
    
    /**
     * Get the total savings amount compared to buying individual packs.
     * 
     * @param ProductPackPrice $packPrice
     * @return float
     */
    public function getTotalSavingsAmount(ProductPackPrice $packPrice)
    {
        if (!$packPrice || $packPrice->per_pack_price <= 0) {
            return 0;
        }
        
        $packRegularPrice = $packPrice->per_pack_price;
        $bulkRegularPrice = $this->per_pack_price;
        
        $savingsPerPack = $packRegularPrice - $bulkRegularPrice;
        $totalSavings = $savingsPerPack * $this->number_of_packs;
        
        return round($totalSavings, 2);
    }
}