<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class RefreshToken extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'refresh_tokens';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'token',
        'token_hash',
        'expires_at',
        'revoked_at',
        'revoked_by',
        'revoked_reason',
        'device_info',
        'ip_address',
        'user_agent',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'token_hash',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'expires_at' => 'datetime',
        'revoked_at' => 'datetime',
        'device_info' => 'array',
    ];

    /**
     * Get the user that owns the refresh token.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the user who revoked this token.
     */
    public function revokedBy()
    {
        return $this->belongsTo(User::class, 'revoked_by');
    }

    /**
     * Scope a query to only include active tokens.
     */
    public function scopeActive($query)
    {
        return $query->where('expires_at', '>', now())
                    ->whereNull('revoked_at');
    }

    /**
     * Scope a query to only include expired tokens.
     */
    public function scopeExpired($query)
    {
        return $query->where('expires_at', '<=', now());
    }

    /**
     * Scope a query to only include revoked tokens.
     */
    public function scopeRevoked($query)
    {
        return $query->whereNotNull('revoked_at');
    }

    /**
     * Check if the token is active (not expired and not revoked).
     */
    public function isActive(): bool
    {
        return $this->expires_at->isFuture() && is_null($this->revoked_at);
    }

    /**
     * Check if the token is expired.
     */
    public function isExpired(): bool
    {
        return $this->expires_at->isPast();
    }

    /**
     * Check if the token is revoked.
     */
    public function isRevoked(): bool
    {
        return !is_null($this->revoked_at);
    }

    /**
     * Revoke the token.
     */
    public function revoke(?User $revokedBy = null, string $reason = 'manual_revocation'): bool
    {
        return $this->update([
            'revoked_at' => now(),
            'revoked_by' => $revokedBy?->id,
            'revoked_reason' => $reason,
        ]);
    }

    /**
     * Create a new refresh token for a user.
     */
    public static function createForUser(User $user, array $deviceInfo = []): self
    {
        $token = self::generateToken();
        
        return self::create([
            'user_id' => $user->id,
            'token' => $token,
            'token_hash' => Hash::make($token),
            'expires_at' => now()->addMinutes(config('auth_settings.token_expiration.refresh_token', 1440)),
            'device_info' => $deviceInfo,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);
    }

    /**
     * Find a refresh token by its token value.
     */
    public static function findByToken(string $token): ?self
    {
        return self::where('token', $token)->first();
    }

    /**
     * Find an active refresh token by its token value.
     */
    public static function findActiveByToken(string $token): ?self
    {
        return self::where('token', $token)
                  ->active()
                  ->first();
    }

    /**
     * Revoke all tokens for a user.
     */
    public static function revokeAllForUser(User $user, ?User $revokedBy = null, string $reason = 'bulk_revocation'): int
    {
        return self::where('user_id', $user->id)
                  ->whereNull('revoked_at')
                  ->update([
                      'revoked_at' => now(),
                      'revoked_by' => $revokedBy?->id,
                      'revoked_reason' => $reason,
                  ]);
    }

    /**
     * Clean up expired tokens.
     */
    public static function cleanupExpired(): int
    {
        return self::expired()->delete();
    }

    /**
     * Clean up revoked tokens older than specified days.
     */
    public static function cleanupRevoked(int $days = 30): int
    {
        return self::revoked()
                  ->where('revoked_at', '<', now()->subDays($days))
                  ->delete();
    }

    /**
     * Get active tokens count for a user.
     */
    public static function getActiveCountForUser(User $user): int
    {
        return self::where('user_id', $user->id)->active()->count();
    }

    /**
     * Check if user has too many active tokens.
     */
    public static function hasTooManyTokens(User $user): bool
    {
        $maxTokens = config('auth_settings.security.max_concurrent_sessions', 5);
        return self::getActiveCountForUser($user) >= $maxTokens;
    }

    /**
     * Generate a unique refresh token.
     */
    private static function generateToken(): string
    {
        do {
            $token = 'rt_' . Str::random(64);
        } while (self::where('token', $token)->exists());

        return $token;
    }

    /**
     * Get the device information as a formatted string.
     */
    public function getDeviceInfoString(): string
    {
        if (empty($this->device_info)) {
            return 'Unknown Device';
        }

        $info = $this->device_info;
        $parts = [];

        if (isset($info['device'])) {
            $parts[] = ucfirst($info['device']);
        }

        if (isset($info['browser'])) {
            $parts[] = ucfirst($info['browser']);
        }

        if (isset($info['os'])) {
            $parts[] = $info['os'];
        }

        return !empty($parts) ? implode(' - ', $parts) : 'Unknown Device';
    }

    /**
     * Get the token's age in human-readable format.
     */
    public function getAgeAttribute(): string
    {
        return $this->created_at->diffForHumans();
    }

    /**
     * Get the time until expiration in human-readable format.
     */
    public function getExpiresInAttribute(): string
    {
        if ($this->isExpired()) {
            return 'Expired';
        }

        return $this->expires_at->diffForHumans();
    }
} 