<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Setting extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'default_customer_margin',
        'default_partner_margin',
        'default_customer_margin_type',
        'default_partner_margin_type',
        'default_currency_code',
        'default_tax_rate',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'default_customer_margin' => 'decimal:2',
        'default_partner_margin' => 'decimal:2',
        'default_tax_rate' => 'decimal:2',
    ];

    /**
     * Get the default currency.
     */
    public function defaultCurrency()
    {
        return $this->belongsTo(Currency::class, 'default_currency_code', 'code');
    }

    /**
     * Boot function from Laravel.
     */
    protected static function boot()
    {
        parent::boot();
        
        // Ensure we have valid defaults when creating settings
        static::creating(function($settings) {
            if (empty($settings->default_customer_margin)) {
                $settings->default_customer_margin = 15.00;
            }
            
            if (empty($settings->default_partner_margin)) {
                $settings->default_partner_margin = 10.00;
            }
            
            if (empty($settings->default_customer_margin_type)) {
                $settings->default_customer_margin_type = 'percentage';
            }
            
            if (empty($settings->default_partner_margin_type)) {
                $settings->default_partner_margin_type = 'percentage';
            }
            
            if (empty($settings->default_currency_code)) {
                $defaultCurrency = Currency::getDefault();
                $settings->default_currency_code = $defaultCurrency ? $defaultCurrency->code : 'AUD';
            }
            
            if (empty($settings->default_tax_rate)) {
                $settings->default_tax_rate = 10.00;
            }
        });
    }

    /**
     * Get the default customer margin as an array with value and type.
     *
     * @return array
     */
    public function getDefaultCustomerMargin()
    {
        return [
            'value' => $this->default_customer_margin,
            'type' => $this->default_customer_margin_type
        ];
    }

    /**
     * Get the default partner margin as an array with value and type.
     *
     * @return array
     */
    public function getDefaultPartnerMargin()
    {
        return [
            'value' => $this->default_partner_margin,
            'type' => $this->default_partner_margin_type
        ];
    }

    /**
     * Get the default tax rate.
     *
     * @return float
     */
    public function getDefaultTaxRate()
    {
        return $this->default_tax_rate ?? 10.00;
    }

    /**
     * Get site-wide settings.
     *
     * @return Setting
     */
    public static function getSiteSettings()
    {
        $settings = static::first();
        
        if (!$settings) {
            // Create default settings if none exist
            $settings = static::create([
                'default_customer_margin' => 15.00,
                'default_partner_margin' => 10.00,
                'default_customer_margin_type' => 'percentage',
                'default_partner_margin_type' => 'percentage',
                'default_currency_code' => Currency::getDefaultCode(),
            ]);
        }
        
        return $settings;
    }

    /**
     * Format a price with the default currency symbol.
     *
     * @param float $amount
     * @return string
     */
    public function formatPrice($amount)
    {
        $currency = $this->defaultCurrency;
        
        if ($currency) {
            return $currency->format($amount);
        }
        
        // Fallback to dollar sign if no currency is set
        return '$' . number_format($amount, 2);
    }
}