<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Product extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'title',
        'slug',
        'description',
        'sku',
        'brand_id',
        'vendor_id',
        'created_by',
        'color_options',
        'storage_options',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'is_approved',
        'approval_status',
        'approved_by',
        'approval_date',
        'rejection_reason',
        'quantity',
        'stock_status',
        'rating_average',
        'rating_count',
        'reviews',
        'fulfilled_by',
        'ships_from',
        'eligible_for_free_shipping',
        'is_featured',
        'is_active',
        'currency_code',
        'default_customer_margin',
        'default_partner_margin',
        'default_customer_margin_type',
        'default_partner_margin_type',
        'min_order_quantity',
        'min_order_value',
        'free_shipping_threshold',
        'products_per_pallet',
        'pallet_delivery_fee',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'color_options' => 'json',
        'storage_options' => 'json',
        'meta_keywords' => 'json',
        'reviews' => 'json',
        'is_approved' => 'boolean',
        'is_featured' => 'boolean',
        'is_active' => 'boolean',
        'eligible_for_free_shipping' => 'boolean',
        'approval_date' => 'datetime',
        'rating_average' => 'decimal:2',
        'rating_count' => 'integer',
        'quantity' => 'integer',
        'default_customer_margin' => 'decimal:2',
        'default_partner_margin' => 'decimal:2',
        'min_order_quantity' => 'integer',
        'min_order_value' => 'decimal:2',
        'free_shipping_threshold' => 'integer',
        'products_per_pallet' => 'integer',
        'pallet_delivery_fee' => 'decimal:2',
    
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        'base_price',
        'current_price',
        'discount_percentage',
    ];

    /**
     * Boot function from Laravel.
     */
    protected static function boot()
    {
        parent::boot();
        
        // When creating a product, automatically generate a slug from the name if not provided
        static::creating(function ($product) {
            if (empty($product->slug)) {
                $product->slug = Str::slug($product->name);
            }
            
            // Set default currency if not provided
            if (empty($product->currency_code)) {
                // Get default currency from settings
                $settings = Setting::first();
                $product->currency_code = $settings ? $settings->default_currency_code : 'AUD';
            }
        });
        
        // When updating, regenerate slug if name changes and slug wasn't manually set
        static::updating(function ($product) {
            if ($product->isDirty('name') && !$product->isDirty('slug')) {
                $product->slug = Str::slug($product->name);
            }
        });
    }

    /**
     * Get the brand that owns the product.
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get the currency for the product.
     */
    public function currency()
    {
        return $this->belongsTo(Currency::class, 'currency_code', 'code');
    }

    /**
     * Get the vendor (user) that owns the product.
     */
    public function vendor()
    {
        return $this->belongsTo(User::class, 'vendor_id');
    }

    /**
     * Get the user who created the product.
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the user who approved the product.
     */
    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Get the categories for the product.
     */
    public function categories()
    {
        return $this->belongsToMany(Category::class, 'product_categories');
    }

    /**
     * Get the images for the product.
     */
    public function images()
    {
        return $this->hasMany(ProductImage::class)->orderBy('sort_order');
    }

    /**
     * Get the variant stocks for the product.
     */
    public function variantStocks()
    {
        return $this->hasMany(ProductVariantStock::class);
    }

    /**
     * Get the pack price for the product.
     */
    public function packPrice()
    {
        return $this->hasOne(ProductPackPrice::class);
    }

    /**
     * Get the bulk prices for the product.
     */
    public function bulkPrices()
    {
        return $this->hasMany(ProductBulkPrice::class)->orderBy('number_of_packs');
    }

    /**
     * Get the offers for the product.
     */
    public function offers()
    {
        return $this->hasMany(ProductOffer::class);
    }

    /**
     * Get active offers for the product.
     */
    public function activeOffers()
    {
        return $this->hasMany(ProductOffer::class)
            ->where('is_active', true)
            ->where(function ($query) {
                $query->whereNull('start_date')
                    ->orWhere('start_date', '<=', now());
            })
            ->where(function ($query) {
                $query->whereNull('end_date')
                    ->orWhere('end_date', '>=', now());
            });
    }

    /**
     * Get base price attribute (pack price).
     */
    public function getBasePriceAttribute()
    {
        $packPrice = $this->packPrice;
        return $packPrice ? $packPrice->per_pack_price : null;
    }

    /**
     * Get current price attribute (considering special price).
     */
    public function getCurrentPriceAttribute()
    {
        $packPrice = $this->packPrice;
        if (!$packPrice) {
            return null;
        }
        
        return $packPrice->per_pack_special_price ?? $packPrice->per_pack_price;
    }

    /**
     * Get discount percentage attribute.
     */
    public function getDiscountPercentageAttribute()
    {
        $packPrice = $this->packPrice;
        if (!$packPrice || !$packPrice->per_pack_special_price) {
            return 0;
        }
        
        $regularPrice = $packPrice->per_pack_price;
        $specialPrice = $packPrice->per_pack_special_price;
        
        if ($regularPrice <= 0) {
            return 0;
        }
        
        return round((($regularPrice - $specialPrice) / $regularPrice) * 100, 2);
    }

    /**
     * Get the default customer margin (from product, or fall back to global setting)
     */
    public function getDefaultCustomerMargin()
    {
        if (!is_null($this->default_customer_margin)) {
            return [
                'value' => $this->default_customer_margin,
                'type' => $this->default_customer_margin_type ?? 'percentage'
            ];
        }
        
        // Fall back to global settings
        $settings = Setting::first();
        if ($settings) {
            return [
                'value' => $settings->default_customer_margin,
                'type' => $settings->default_customer_margin_type
            ];
        }
        
        // Default fallback
        return [
            'value' => 15.00,
            'type' => 'percentage'
        ];
    }

    /**
     * Get the default partner margin (from product, or fall back to global setting)
     */
    public function getDefaultPartnerMargin()
    {
        if (!is_null($this->default_partner_margin)) {
            return [
                'value' => $this->default_partner_margin,
                'type' => $this->default_partner_margin_type ?? 'percentage'
            ];
        }
        
        // Fall back to global settings
        $settings = Setting::first();
        if ($settings) {
            return [
                'value' => $settings->default_partner_margin,
                'type' => $settings->default_partner_margin_type
            ];
        }
        
        // Default fallback
        return [
            'value' => 10.00,
            'type' => 'percentage'
        ];
    }

    /**
     * Format price with currency symbol
     */
    public function formatPrice($price)
    {
        $currency = $this->currency;
        $symbol = $currency ? $currency->symbol : '$';
        
        return $symbol . number_format($price, 2);
    }

    /**
     * Scope a query to only include active products.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include featured products.
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Scope a query to only include approved products.
     */
    public function scopeApproved($query)
    {
        return $query->where('is_approved', true);
    }

    /**
     * Scope a query to only include products that are in stock.
     */
    public function scopeInStock($query)
    {
        return $query->where('stock_status', 'in_stock');
    }

    /**
     * Scope a query to filter products by brand.
     */
    public function scopeByBrand($query, $brandId)
    {
        return $query->where('brand_id', $brandId);
    }

    /**
     * Scope a query to filter products by vendor.
     */
    public function scopeByVendor($query, $vendorId)
    {
        return $query->where('vendor_id', $vendorId);
    }

    /**
     * Scope a query to filter products by currency.
     */
    public function scopeByCurrency($query, $currencyCode)
    {
        return $query->where('currency_code', $currencyCode);
    }

    /**
     * Scope a query to filter products by price range.
     */
    public function scopeByPriceRange($query, $min, $max)
    {
        return $query->whereHas('packPrice', function ($q) use ($min, $max) {
            $q->where(function ($subQuery) use ($min, $max) {
                $subQuery->where(function ($priceQuery) use ($min, $max) {
                    $priceQuery->whereNull('per_pack_special_price')
                        ->where('per_pack_price', '>=', $min)
                        ->where('per_pack_price', '<=', $max);
                })->orWhere(function ($specialPriceQuery) use ($min, $max) {
                    $specialPriceQuery->whereNotNull('per_pack_special_price')
                        ->where('per_pack_special_price', '>=', $min)
                        ->where('per_pack_special_price', '<=', $max);
                });
            });
        });
    }

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    /**
     * Scope a query to filter products by category.
     */
    public function scopeByCategory($query, $categoryId)
    {
        return $query->whereHas('categories', function ($q) use ($categoryId) {
            $q->where('categories.id', $categoryId);
        });
    }

    /**
     * Scope a query to search products by keyword.
     */
    public function scopeSearch($query, $keyword)
    {
        return $query->where(function ($q) use ($keyword) {
            $q->where('name', 'like', '%' . $keyword . '%')
                ->orWhere('title', 'like', '%' . $keyword . '%')
                ->orWhere('description', 'like', '%' . $keyword . '%')
                ->orWhere('sku', 'like', '%' . $keyword . '%');
        });
    }
}