<?php

namespace App\Services;

use App\Models\Cart;

class CartService
{
    /**
     * Helper method to determine which margin to use based on user role
     */
    private function getMarginForUser($cartItem, $user)
    {
        if ($user && ($user->hasRole('partner') || $user->hasRole('vendor'))) {
            return [
                'value' => $cartItem->partner_margin,
                'type' => $cartItem->partner_margin_type
            ];
        }
        
        return [
            'value' => $cartItem->customer_margin,
            'type' => $cartItem->customer_margin_type
        ];
    }

    /**
     * Get a summary of the cart, including all pricing breakdowns and items.
     *
     * @param Cart $cart
     * @param \App\Models\User|null $user
     * @return array
     */
    public function getCartSummary(Cart $cart, $user = null)
    {
        $cart->loadMissing([
            'items.product.brand',
            'items.product.images',
            'items.bulkPrice',
            'items.packPrice',
            'currency',
            'appliedMembership'
        ]);

        // Load user membership if not provided
        if ($user) {
            $user->load(['customerProfile.membership']);
        }

        $formattedItems = $cart->items->map(function ($item) use ($user) {
            $item->loadMissing(['packPrice', 'bulkPrice']);
            $base_unit_price_with_margin = null;
            $special_unit_price_with_margin = null;
            $subtotal = null;
            $quantity = $item->quantity;
            
            // Get appropriate margin based on user role
            $marginData = $this->getMarginForUser($item, $user);
            $margin = $marginData['value'];
            $marginType = $marginData['type'];

            // Use the actual unit_price that was set when the cart item was created
            $basePrice = $item->unit_price;
            
            if ($marginType === 'percentage') {
                $base_unit_price_with_margin = round($basePrice * (1 + ($margin / 100)), 2);
                // For special pricing, we need to check if there's a special price available
                if ($item->is_bulk_pricing && $item->bulkPrice && $item->bulkPrice->per_pack_special_price) {
                    $special_unit_price_with_margin = round($item->bulkPrice->per_pack_special_price * (1 + ($margin / 100)), 2);
                } elseif ($item->is_pack_pricing && $item->packPrice && $item->packPrice->per_pack_special_price) {
                    $special_unit_price_with_margin = round($item->packPrice->per_pack_special_price * (1 + ($margin / 100)), 2);
                } else {
                    $special_unit_price_with_margin = null;
                }
            } else {
                $base_unit_price_with_margin = round($basePrice + $margin, 2);
                // For special pricing, we need to check if there's a special price available
                if ($item->is_bulk_pricing && $item->bulkPrice && $item->bulkPrice->per_pack_special_price) {
                    $special_unit_price_with_margin = round($item->bulkPrice->per_pack_special_price + $margin, 2);
                } elseif ($item->is_pack_pricing && $item->packPrice && $item->packPrice->per_pack_special_price) {
                    $special_unit_price_with_margin = round($item->packPrice->per_pack_special_price + $margin, 2);
                } else {
                    $special_unit_price_with_margin = null;
                }
            }

            if ($special_unit_price_with_margin !== null) {
                $subtotal = round($special_unit_price_with_margin * $quantity, 2);
            } else {
                $subtotal = round($base_unit_price_with_margin * $quantity, 2);
            }

            // Calculate membership discount for this item
            $membershipDiscount = $this->calculateItemMembershipDiscount($subtotal, $user);

            // Tax rates visibility
            $effective_tax_rate = $item->getTaxRate();
            $product_tax_rate = $item->product ? $item->product->getTaxRate() : null;
            $bulk_tax_rate = ($item->is_bulk_pricing && $item->bulkPrice) ? $item->bulkPrice->tax_rate : null;
            $pack_tax_rate = ($item->is_pack_pricing && $item->packPrice) ? $item->packPrice->tax_rate : null;

            return [
                'id' => $item->id,
                'cart_id' => $item->cart_id,
                'product_id' => $item->product_id,
                'quantity' => $quantity,
                'unit_price' => $item->unit_price,
                'subtotal' => $subtotal,
                'discount_amount' => $item->discount_amount,
                'membership_discount_amount' => $membershipDiscount['amount'],
                'membership_discount_percentage' => $membershipDiscount['percentage'],
                'price_after_membership_discount' => $subtotal - $membershipDiscount['amount'],
                'effective_tax_rate' => $effective_tax_rate,
                'product_tax_rate' => $product_tax_rate,
                'bulk_tax_rate' => $bulk_tax_rate,
                'pack_tax_rate' => $pack_tax_rate,
                'is_bulk_pricing' => $item->is_bulk_pricing,
                'is_pack_pricing' => $item->is_pack_pricing,
                'notes' => $item->notes,
                'customer_margin' => $item->customer_margin,
                'partner_margin' => $item->partner_margin,
                'customer_margin_type' => $item->customer_margin_type,
                'partner_margin_type' => $item->partner_margin_type,
                'applied_margin' => $margin,
                'applied_margin_type' => $marginType,
                'created_at' => $item->created_at,
                'updated_at' => $item->updated_at,
                'formatted_quantity' => $item->formatted_quantity,
                'base_unit_price_with_margin' => $base_unit_price_with_margin,
                'special_unit_price_with_margin' => $special_unit_price_with_margin,
                'bulk_price' => $item->bulk_price_id ? \App\Models\ProductBulkPrice::find($item->bulk_price_id) : null,
                'pack_price' => $item->pack_price_id ? \App\Models\ProductPackPrice::find($item->pack_price_id) : null,
                'delivery_fee' => $item->delivery_fee,
                'product' => [
                    'id' => $item->product->id,
                    'name' => $item->product->name,
                    'title' => $item->product->title,
                    'sku' => $item->product->sku,
                    'tax_rate' => $product_tax_rate,
                    'quantity' => $item->product->quantity,
                    'stock_status' => $item->product->stock_status,
                    'images' => $item->product->images
                ]
            ];
        });

        $cart_total_amount = array_sum(array_column($formattedItems->toArray(), 'subtotal'));
        
        // Calculate tax per item based on product-specific tax rates
        $tax_amount = 0;
        foreach ($cart->items as $item) {
            $taxRate = $item->getTaxRate();
            $taxableAmount = $item->subtotal - $item->discount_amount;
            $tax_amount += round($taxableAmount * ($taxRate / 100), 2);
        }
        
        $total_delivery_fees = array_sum(array_column($formattedItems->toArray(), 'delivery_fee'));
        
        // Calculate membership discount on total_amount (subtotal)
        $membershipDiscount = $cart->calculateMembershipDiscount($user);
        $membership_discount_amount = $membershipDiscount['amount'];
        $membership_discount_percentage = $membershipDiscount['percentage'];
        
        // Apply membership discount to total_amount, then calculate final price
        $total_amount_after_membership_discount = $cart_total_amount - $membership_discount_amount;
        
        // Recalculate tax after membership discount (proportional to each item's tax rate)
        $tax_amount_after_membership_discount = 0;
        foreach ($cart->items as $item) {
            $taxRate = $item->getTaxRate();
            $itemSubtotal = $item->subtotal - $item->discount_amount;
            $itemMembershipDiscount = $this->calculateItemMembershipDiscount($itemSubtotal, $user)['amount'];
            $taxableAmountAfterDiscount = $itemSubtotal - $itemMembershipDiscount;
            $tax_amount_after_membership_discount += round($taxableAmountAfterDiscount * ($taxRate / 100), 2);
        }
        $total_amount_with_tax = round($total_amount_after_membership_discount + $tax_amount_after_membership_discount, 2);
        $final_price = round($total_amount_after_membership_discount + $tax_amount_after_membership_discount + $total_delivery_fees, 2);

        return [
            'id' => $cart->id,
            'user_id' => $cart->user_id,
            'session_id' => $cart->session_id,
            'notes' => $cart->notes,
            'discount_amount' => round($cart->discount_amount + $membership_discount_amount, 2),
            'tax_amount' => $tax_amount_after_membership_discount,
            'total_amount' => $total_amount_after_membership_discount,
            'currency_code' => $cart->currency_code,
            'is_active' => $cart->is_active,
            'expires_at' => $cart->expires_at,
            'created_at' => $cart->created_at,
            'updated_at' => $cart->updated_at,
            'total_amount_with_tax' => $total_amount_with_tax,
            'total_delivery_fees' => $total_delivery_fees,
            'final_price' => $final_price,
            'membership_info' => $user ? $user->getMembershipDiscountInfo() : null,
            'membership_discount_amount' => $membership_discount_amount,
            'membership_discount_percentage' => $membership_discount_percentage,
            'items' => $formattedItems,
            'currency' => $cart->currency
        ];
    }

    /**
     * Calculate membership discount for a specific item.
     */
    private function calculateItemMembershipDiscount($itemSubtotal, $user)
    {
        if (!$user) return ['amount' => 0, 'percentage' => 0];
        
        $membership = $user->getActiveMembership();
        if (!$membership) return ['amount' => 0, 'percentage' => 0];
        
        $discountPercentage = $membership->extra_discount;
        $discountAmount = ($itemSubtotal * $discountPercentage) / 100;
        
        return [
            'amount' => round($discountAmount, 2),
            'percentage' => $discountPercentage
        ];
    }
} 