<?php

namespace App\Services;

use SendGrid;
use SendGrid\Mail\Mail;
use Illuminate\Support\Facades\Log;

class EmailService
{
    protected $sendGrid;
    protected $fromEmail;
    protected $fromName;

    public function __construct()
    {
        $this->sendGrid = new SendGrid(config('services.sendgrid.key'));
        $this->fromEmail = config('mail.from.address');
        $this->fromName = config('mail.from.name');
    }

    /**
     * Send an email using SendGrid
     *
     * @param string $to
     * @param string $subject
     * @param string $content
     * @param array $attachments
     * @return bool
     */
    public function send($to, $subject, $content, $attachments = [])
    {
        try {
            $email = new Mail();
            $email->setFrom($this->fromEmail, $this->fromName);
            $email->setSubject($subject);
            $email->addTo($to);
            $email->addContent("text/html", $content);

            // Add attachments if any
            foreach ($attachments as $attachment) {
                $email->addAttachment(
                    $attachment['content'],
                    $attachment['type'],
                    $attachment['filename'],
                    $attachment['disposition'] ?? 'attachment'
                );
            }

            $response = $this->sendGrid->send($email);

            if ($response->statusCode() >= 200 && $response->statusCode() < 300) {
                Log::info('Email sent successfully', [
                    'to' => $to,
                    'subject' => $subject,
                    'status' => $response->statusCode()
                ]);
                return true;
            }

            Log::error('Failed to send email', [
                'to' => $to,
                'subject' => $subject,
                'status' => $response->statusCode(),
                'body' => $response->body()
            ]);
            return false;

        } catch (\Exception $e) {
            Log::error('Exception while sending email', [
                'to' => $to,
                'subject' => $subject,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send a test email
     *
     * @param string $to
     * @return bool
     */
    public function sendTestEmail($to)
    {
        $subject = 'Test Email from B2B E-commerce API';
        $content = '
            <h1>Test Email</h1>
            <p>This is a test email sent from your B2B E-commerce API.</p>
            <p>If you received this email, your email configuration is working correctly.</p>
            <p>Time sent: ' . now() . '</p>
        ';

        return $this->send($to, $subject, $content);
    }
} 