<?php

namespace App\Services;

use App\Models\Order;
use App\Models\Cart;
use App\Models\CartItem;
use App\Models\OrderItem;
use App\Models\Invoice;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class OrderService
{
    /**
     * Get cart with all necessary relationships
     *
     * @param int $cartId
     * @return Cart|null
     */
    public function getCart($cartId)
    {
        return Cart::with([
            'items.product.brand',
            'items.product.images',
            'items.packPrice',
            'items.bulkPrice',
            'currency'
        ])
        ->where('id', $cartId)
        ->where('user_id', auth()->id())
        ->first();
    }

    /**
     * Calculate total amount for a cart
     *
     * @param Cart $cart
     * @return float
     */
    public function calculateTotal(Cart $cart)
    {
        $subtotal = round($cart->items->sum('subtotal'), 2);
        $discount = round($cart->items->sum('discount_amount'), 2);
        $tax = round($cart->items->sum('tax_amount'), 2);
        $shipping = round($cart->shipping_amount ?? 0, 2);

        return round($subtotal - $discount + $tax + $shipping, 2);
    }

    /**
     * Calculate final price using the same logic as cart display
     * This ensures consistency between cart display and payment amount
     *
     * @param Cart $cart
     * @return float
     */
    public function calculateFinalPrice(Cart $cart)
    {
        $user = auth()->user();
        $user->load(['role', 'customerProfile.membership']); // Load membership
        
        // Format items using the same logic as CartController::index()
        $formattedItems = $cart->items->map(function ($item) use ($user) {
            $base_unit_price_with_margin = null;
            $special_unit_price_with_margin = null;
            $subtotal = null;
            $quantity = $item->quantity;
            
            // Get appropriate margin based on user role
            if ($user && ($user->hasRole('partner') || $user->hasRole('vendor'))) {
                $margin = $item->partner_margin;
                $marginType = $item->partner_margin_type;
            } else {
                $margin = $item->customer_margin;
                $marginType = $item->customer_margin_type;
            }

            // Use the actual unit_price that was set when the cart item was created
            $base_price = $item->unit_price;
            
            // Check for special pricing
            if ($item->is_bulk_pricing && $item->bulkPrice && $item->bulkPrice->per_pack_special_price) {
                $special_price = $item->bulkPrice->per_pack_special_price;
            } elseif ($item->is_pack_pricing && $item->packPrice && $item->packPrice->per_pack_special_price) {
                $special_price = $item->packPrice->per_pack_special_price;
            } else {
                $special_price = null;
            }

            if ($marginType === 'percentage') {
                $base_unit_price_with_margin = round($base_price * (1 + ($margin / 100)), 2);
                if ($special_price && $special_price > 0) {
                    $special_unit_price_with_margin = round($special_price * (1 + ($margin / 100)), 2);
                }
            } else {
                $base_unit_price_with_margin = round($base_price + $margin, 2);
                if ($special_price && $special_price > 0) {
                    $special_unit_price_with_margin = round($special_price + $margin, 2);
                }
            }

            if ($special_unit_price_with_margin !== null) {
                $subtotal = round($special_unit_price_with_margin * $quantity, 2);
            } else {
                $subtotal = round($base_unit_price_with_margin * $quantity, 2);
            }

            return [
                'product_id' => $item->product_id,
                'quantity' => $quantity,
                'pack_price_id' => $item->pack_price_id,
                'bulk_price_id' => $item->bulk_price_id,
                'subtotal' => $subtotal,
                'delivery_fee' => $item->delivery_fee
            ];
        })->toArray();

        // Use the same fillCartItemPricingDetails logic as CartController
        list($formattedItems, $deliveryFees) = $this->fillCartItemPricingDetails($formattedItems);

        // Calculate totals using the same logic as CartController
        $cart_total_amount = array_sum(array_column($formattedItems, 'subtotal'));
        
        // Calculate tax per item based on product-specific tax rates
        $tax_amount = 0;
        foreach ($cart->items as $item) {
            $taxRate = $item->getTaxRate();
            $taxableAmount = $item->subtotal - $item->discount_amount;
            $tax_amount += round($taxableAmount * ($taxRate / 100), 2);
        }
        
        $total_delivery_fees = array_sum($deliveryFees);
        
        // Calculate membership discount on total_amount (subtotal)
        $membershipDiscount = $cart->calculateMembershipDiscount($user);
        $membership_discount_amount = $membershipDiscount['amount'];
        
        // Apply membership discount to total_amount, then calculate final price
        $total_amount_after_membership_discount = $cart_total_amount - $membership_discount_amount;
        
        // Recalculate tax after membership discount (proportional to each item's tax rate)
        $tax_amount_after_membership_discount = 0;
        foreach ($cart->items as $item) {
            $taxRate = $item->getTaxRate();
            $itemSubtotal = $item->subtotal - $item->discount_amount;
            $itemMembershipDiscount = $this->calculateItemMembershipDiscount($itemSubtotal, $user)['amount'];
            $taxableAmountAfterDiscount = $itemSubtotal - $itemMembershipDiscount;
            $tax_amount_after_membership_discount += round($taxableAmountAfterDiscount * ($taxRate / 100), 2);
        }
        $total_amount_with_tax = round($total_amount_after_membership_discount + $tax_amount_after_membership_discount, 2);
        $final_price = round($total_amount_after_membership_discount + $tax_amount_after_membership_discount + $total_delivery_fees, 2);
        
        return $final_price;
    }

    /**
     * Calculate membership discount for a specific item.
     */
    private function calculateItemMembershipDiscount($itemSubtotal, $user)
    {
        if (!$user) return ['amount' => 0, 'percentage' => 0];
        
        $membership = $user->getActiveMembership();
        if (!$membership) return ['amount' => 0, 'percentage' => 0];
        
        $discountPercentage = $membership->extra_discount;
        $discountAmount = ($itemSubtotal * $discountPercentage) / 100;
        
        return [
            'amount' => round($discountAmount, 2),
            'percentage' => $discountPercentage
        ];
    }

    /**
     * Fill cart item pricing details - same logic as CartController
     */
    public function fillCartItemPricingDetails($items)
    {
        $deliveryFees = [];
        foreach ($items as $idx => &$item) {
            $deliveryFee = 0;
            $bulkPrice = null;
            $packPrice = null;

            // Fetch all bulk prices for the product, ordered by number_of_packs ASC
            $bulkPrices = \DB::table('product_bulk_prices')
                ->where('product_id', $item['product_id'])
                ->orderBy('number_of_packs', 'asc')
                ->get();

            // Find the best bulk price tier for the quantity
            foreach ($bulkPrices as $bp) {
                if ($item['quantity'] >= $bp->number_of_packs) {
                    $bulkPrice = $bp;
                } else {
                    break;
                }
            }

            // If no bulk price, try pack price
            if (!$bulkPrice && !empty($item['pack_price_id'])) {
                $packPrice = \DB::table('product_pack_prices')->where('id', $item['pack_price_id'])->first();
            }

            $item['bulk_price'] = $bulkPrice;
            $item['pack_price'] = $packPrice;

            if ($bulkPrice && isset($bulkPrice->delivery_fee)) {
                $deliveryFee = $bulkPrice->delivery_fee;
            } elseif ($packPrice && isset($packPrice->delivery_fee)) {
                $deliveryFee = $packPrice->delivery_fee;
            }
            $item['delivery_fee'] = $deliveryFee;
            $deliveryFees[$idx] = $deliveryFee;
        }
        return [$items, $deliveryFees];
    }

    /**
     * Create a new order from cart data
     *
     * @param array $orderData
     * @return Order
     */
    public function createOrder(array $orderData)
    {
        DB::beginTransaction();
        
        try {
            // Create the order
            $order = new Order();
            $order->order_number = 'ORD-' . strtoupper(Str::random(8));
            $order->user_id = auth()->id();
            $order->cart_id = $orderData['cart_id'];
            $order->status = 'pending';
            $order->shipping_address = $orderData['shipping_address'];
            $order->billing_address = $orderData['billing_address'];
            $order->shipping_phone = $orderData['shipping_phone'];
            $order->billing_phone = $orderData['billing_phone'];
            $order->billing_email = $orderData['billing_email'];
            $order->payment_intent_id = $orderData['payment_intent_id'];
            $order->payment_status = $orderData['payment_status'];
            $order->notes = $orderData['notes'] ?? null;
            $order->total_amount = $orderData['total_amount'];
            $order->currency = $orderData['currency'];
            $order->save();

            // Get cart items
            $cart = Cart::with('items.product')->find($orderData['cart_id']);
            
            // Create order items
            foreach ($cart->items as $cartItem) {
                $orderItem = new OrderItem();
                $orderItem->order_id = $order->id;
                $orderItem->product_id = $cartItem->product_id;
                $orderItem->product_name = $cartItem->product->name;
                $orderItem->product_sku = $cartItem->product->sku;
                $orderItem->quantity = $cartItem->quantity;
                $orderItem->unit_price = $cartItem->unit_price;
                $orderItem->subtotal = $cartItem->subtotal;
                $orderItem->discount_amount = $cartItem->discount_amount;
                $orderItem->tax_amount = $cartItem->tax_amount;
                $orderItem->total_amount = $cartItem->total_amount;
                $orderItem->is_bulk_pricing = $cartItem->is_bulk_pricing;
                $orderItem->is_pack_pricing = $cartItem->is_pack_pricing;
                $orderItem->bulk_price_id = $cartItem->bulk_price_id;
                $orderItem->pack_price_id = $cartItem->pack_price_id;
                $orderItem->customer_margin = $cartItem->customer_margin;
                $orderItem->partner_margin = $cartItem->partner_margin;
                $orderItem->customer_margin_type = $cartItem->customer_margin_type;
                $orderItem->partner_margin_type = $cartItem->partner_margin_type;
                $orderItem->save();

                // Update product inventory
                $product = $cartItem->product;
                $product->quantity -= $cartItem->quantity;
                $product->save();
            }

            // Create order history
            $order->history()->create([
                'status' => 'pending',
                'comment' => 'Order created',
                'user_id' => auth()->id()
            ]);

            // Create invoice
            $this->createInvoice($order);

            DB::commit();
            return $order;

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Create an invoice for the order
     *
     * @param Order $order
     * @return Invoice
     */
    public function createInvoice(Order $order)
    {
        $invoice = new Invoice();
        $invoice->order_id = $order->id;
        $invoice->invoice_number = 'INV-' . strtoupper(Str::random(8));
        $invoice->amount = $order->total_amount;
        $invoice->currency = $order->currency;
        $invoice->status = 'pending';
        $invoice->save();

        return $invoice;
    }

    /**
     * Clear the cart after successful order
     *
     * @param int $cartId
     * @return void
     */
    public function clearCart($cartId)
    {
        $cart = Cart::find($cartId);
        if ($cart) {
            // Delete all cart items
            CartItem::where('cart_id', $cart->id)->delete();
            
            // Reset cart totals
            $cart->subtotal = 0;
            $cart->discount_amount = 0;
            $cart->tax_amount = 0;
            $cart->shipping_amount = 0;
            $cart->total_amount = 0;
            $cart->save();
        }
    }
} 