<?php

namespace App\Services;

use App\Models\Product;
use App\Models\Category;
use App\Models\Brand;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class SearchService
{
    /**
     * Perform a universal search across multiple models
     *
     * @param string $query
     * @param array $filters
     * @return array
     */
    public function search($query, $filters = [])
    {
        try {
            Log::info('Starting universal search', ['query' => $query, 'filters' => $filters]);

            $results = [
                'products' => $this->searchProducts($query, $filters),
                'categories' => $this->searchCategories($query, $filters),
                'brands' => $this->searchBrands($query, $filters),
            ];

            // Add total count
            $results['total'] = collect($results)->sum(function ($items) {
                return $items['total'] ?? 0;
            });

            Log::info('Search completed successfully', ['total_results' => $results['total']]);

            return $results;
        } catch (\Exception $e) {
            Log::error('Error in universal search', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]);
            throw $e;
        }
    }

    /**
     * Search products
     *
     * @param string $query
     * @param array $filters
     * @return array
     */
    protected function searchProducts($query, $filters)
    {
        try {
            Log::info('Starting product search', ['query' => $query, 'filters' => $filters]);

            $searchQuery = Product::query()
                ->with(['brand', 'categories', 'images', 'packPrice'])
                ->where(function ($q) use ($query) {
                    $q->where('name', 'like', "%{$query}%")
                      ->orWhere('description', 'like', "%{$query}%")
                      ->orWhere('sku', 'like', "%{$query}%")
                      ->orWhere('meta_title', 'like', "%{$query}%")
                      ->orWhere('meta_description', 'like', "%{$query}%")
                      ->orWhere('meta_keywords', 'like', "%{$query}%")
                      ->orWhereHas('brand', function ($brandQuery) use ($query) {
                          $brandQuery->where('name', 'like', "%{$query}%");
                      })
                      ->orWhereHas('categories', function ($categoryQuery) use ($query) {
                          $categoryQuery->where('name', 'like', "%{$query}%");
                      });
                });

            // Apply filters
            if (!empty($filters['brand_name'])) {
                $searchQuery->whereHas('brand', function ($brandQuery) use ($filters) {
                    $brandQuery->where('name', 'like', "%{$filters['brand_name']}%");
                });
            }

            if (!empty($filters['category_name'])) {
                $searchQuery->whereHas('categories', function ($categoryQuery) use ($filters) {
                    $categoryQuery->where('name', 'like', "%{$filters['category_name']}%");
                });
            }

            // Price range filter using packPrice relationship
            if (!empty($filters['min_price']) || !empty($filters['max_price'])) {
                $searchQuery->whereHas('packPrice', function ($priceQuery) use ($filters) {
                    if (!empty($filters['min_price'])) {
                        $priceQuery->where('per_pack_price', '>=', $filters['min_price']);
                    }
                    if (!empty($filters['max_price'])) {
                        $priceQuery->where('per_pack_price', '<=', $filters['max_price']);
                    }
                });
            }

            if (!empty($filters['in_stock'])) {
                $searchQuery->where('quantity', '>', 0);
            }

            // Get total count
            $total = $searchQuery->count();

            // Apply pagination
            $perPage = $filters['per_page'] ?? 10;
            $page = $filters['page'] ?? 1;
            
            $products = $searchQuery->skip(($page - 1) * $perPage)
                                   ->take($perPage)
                                   ->get();

            Log::info('Product search completed', ['total' => $total]);

            return [
                'data' => $products,
                'total' => $total,
                'per_page' => $perPage,
                'current_page' => $page,
                'last_page' => ceil($total / $perPage)
            ];
        } catch (\Exception $e) {
            Log::error('Error in product search', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]);
            throw $e;
        }
    }

    /**
     * Search categories
     *
     * @param string $query
     * @param array $filters
     * @return array
     */
    protected function searchCategories($query, $filters)
    {
        try {
            Log::info('Starting category search', ['query' => $query]);

            $searchQuery = Category::query()
                ->where(function ($q) use ($query) {
                    $q->where('name', 'like', "%{$query}%")
                      ->orWhere('description', 'like', "%{$query}%")
                      ->orWhere('slug', 'like', "%{$query}%");
                });

            // Get total count
            $total = $searchQuery->count();

            // Apply pagination
            $perPage = $filters['per_page'] ?? 10;
            $page = $filters['page'] ?? 1;
            
            $categories = $searchQuery->skip(($page - 1) * $perPage)
                                     ->take($perPage)
                                     ->get();

            Log::info('Category search completed', ['total' => $total]);

            return [
                'data' => $categories,
                'total' => $total,
                'per_page' => $perPage,
                'current_page' => $page,
                'last_page' => ceil($total / $perPage)
            ];
        } catch (\Exception $e) {
            Log::error('Error in category search', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]);
            throw $e;
        }
    }

    /**
     * Search brands
     *
     * @param string $query
     * @param array $filters
     * @return array
     */
    protected function searchBrands($query, $filters)
    {
        try {
            Log::info('Starting brand search', ['query' => $query]);

            $searchQuery = Brand::query()
                ->where(function ($q) use ($query) {
                    $q->where('name', 'like', "%{$query}%")
                      ->orWhere('description', 'like', "%{$query}%")
                      ->orWhere('slug', 'like', "%{$query}%");
                });

            // Get total count
            $total = $searchQuery->count();

            // Apply pagination
            $perPage = $filters['per_page'] ?? 10;
            $page = $filters['page'] ?? 1;
            
            $brands = $searchQuery->skip(($page - 1) * $perPage)
                                 ->take($perPage)
                                 ->get();

            Log::info('Brand search completed', ['total' => $total]);

            return [
                'data' => $brands,
                'total' => $total,
                'per_page' => $perPage,
                'current_page' => $page,
                'last_page' => ceil($total / $perPage)
            ];
        } catch (\Exception $e) {
            Log::error('Error in brand search', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]);
            throw $e;
        }
    }
} 