<?php

namespace App\Services;

use Stripe\Stripe;
use Stripe\PaymentIntent;
use Stripe\Exception\ApiErrorException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Config;

class StripeService
{
    public function __construct()
    {
        Stripe::setApiKey(Config::get('services.stripe.secret'));
    }

    /**
     * Create a payment intent
     *
     * @param float $amount
     * @param string $currency
     * @param array $metadata
     * @return array
     */
    public function createPaymentIntent($amount, $currency = 'AUD', $metadata = [])
    {
        try {
            Log::info('Creating payment intent', [
                'amount' => $amount,
                'currency' => $currency,
                'metadata' => $metadata
            ]);

            $paymentIntent = PaymentIntent::create([
                'amount' => $amount * 100, // Convert to cents
                'currency' => $currency,
                'metadata' => $metadata,
                'automatic_payment_methods' => [
                    'enabled' => true,
                ],
            ]);

            Log::info('Payment intent created', [
                'payment_intent_id' => $paymentIntent->id,
                'amount' => $paymentIntent->amount,
                'currency' => $paymentIntent->currency,
                'status' => $paymentIntent->status
            ]);

            return [
                'success' => true,
                'client_secret' => $paymentIntent->client_secret,
                'payment_intent_id' => $paymentIntent->id
            ];
        } catch (\Exception $e) {
            Log::error('Error creating payment intent', [
                'error' => $e->getMessage(),
                'amount' => $amount,
                'currency' => $currency
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Retrieve and verify a payment intent
     *
     * @param string $paymentIntentId
     * @return array
     */
    public function retrievePaymentIntent($paymentIntentId)
    {
        try {
            $paymentIntent = PaymentIntent::retrieve($paymentIntentId);
            
            // Return the full payment intent object for internal testing
            return [
                'success' => true,
                'payment_intent' => $paymentIntent,
                'amount' => round($paymentIntent->amount / 100, 2), // Round to 2 decimal places
                'currency' => $paymentIntent->currency,
                'status' => $paymentIntent->status
            ];
        } catch (ApiErrorException $e) {
            Log::error('Stripe payment intent retrieval failed', [
                'error' => $e->getMessage(),
                'payment_intent_id' => $paymentIntentId
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Verify if a payment is successful
     *
     * @param string $paymentIntentId
     * @return array
     */
    public function verifyPayment($paymentIntentId)
    {
        try {
            Log::info('Retrieving payment intent from Stripe', [
                'payment_intent_id' => $paymentIntentId
            ]);

            $paymentIntent = PaymentIntent::retrieve($paymentIntentId);
            
            Log::info('Payment intent retrieved', [
                'payment_intent_id' => $paymentIntentId,
                'status' => $paymentIntent->status,
                'amount' => $paymentIntent->amount,
                'currency' => $paymentIntent->currency,
                'last_payment_error' => $paymentIntent->last_payment_error ? [
                    'code' => $paymentIntent->last_payment_error->code,
                    'message' => $paymentIntent->last_payment_error->message,
                    'type' => $paymentIntent->last_payment_error->type
                ] : null,
                'charges' => $paymentIntent->charges ? [
                    'total_count' => $paymentIntent->charges->total_count,
                    'data' => array_map(function($charge) {
                        return [
                            'id' => $charge->id,
                            'status' => $charge->status,
                            'amount' => $charge->amount,
                            'currency' => $charge->currency,
                            'failure_code' => $charge->failure_code,
                            'failure_message' => $charge->failure_message
                        ];
                    }, $paymentIntent->charges->data)
                ] : null
            ]);
            
            // Check if payment is successful
            if ($paymentIntent->status === 'succeeded') {
                return [
                    'success' => true,
                    'amount' => round($paymentIntent->amount / 100, 2),
                    'currency' => $paymentIntent->currency,
                    'status' => $paymentIntent->status
                ];
            }

            // Get detailed error information
            $errorDetails = [
                'status' => $paymentIntent->status,
                'last_payment_error' => $paymentIntent->last_payment_error ? [
                    'code' => $paymentIntent->last_payment_error->code,
                    'message' => $paymentIntent->last_payment_error->message,
                    'type' => $paymentIntent->last_payment_error->type
                ] : null
            ];

            if ($paymentIntent->charges && $paymentIntent->charges->data) {
                $lastCharge = end($paymentIntent->charges->data);
                $errorDetails['last_charge'] = [
                    'status' => $lastCharge->status,
                    'failure_code' => $lastCharge->failure_code,
                    'failure_message' => $lastCharge->failure_message
                ];
            }

            Log::error('Payment verification failed', [
                'payment_intent_id' => $paymentIntentId,
                'error_details' => $errorDetails
            ]);

            return [
                'success' => false,
                'error' => 'Payment not successful',
                'status' => $paymentIntent->status,
                'error_details' => $errorDetails
            ];
        } catch (ApiErrorException $e) {
            Log::error('Stripe API error during payment verification', [
                'error' => $e->getMessage(),
                'error_type' => get_class($e),
                'error_code' => $e->getStripeCode(),
                'http_status' => $e->getHttpStatus(),
                'payment_intent_id' => $paymentIntentId,
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
                'error_code' => $e->getStripeCode(),
                'http_status' => $e->getHttpStatus()
            ];
        } catch (\Exception $e) {
            Log::error('Unexpected error during payment verification', [
                'error' => $e->getMessage(),
                'error_type' => get_class($e),
                'payment_intent_id' => $paymentIntentId,
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Confirm a payment intent
     *
     * @param string $paymentIntentId
     * @return array
     */
    public function confirmPaymentIntent($paymentIntentId)
    {
        try {
            $paymentIntent = PaymentIntent::confirm($paymentIntentId);

            return [
                'success' => true,
                'payment_intent' => $paymentIntent
            ];
        } catch (ApiErrorException $e) {
            Log::error('Stripe payment intent confirmation failed', [
                'error' => $e->getMessage(),
                'payment_intent_id' => $paymentIntentId
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Cancel a payment intent
     *
     * @param string $paymentIntentId
     * @return array
     */
    public function cancelPaymentIntent($paymentIntentId)
    {
        try {
            $paymentIntent = PaymentIntent::cancel($paymentIntentId);

            return [
                'success' => true,
                'payment_intent' => $paymentIntent
            ];
        } catch (ApiErrorException $e) {
            Log::error('Stripe payment intent cancellation failed', [
                'error' => $e->getMessage(),
                'payment_intent_id' => $paymentIntentId
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Confirm a payment intent with a payment method
     *
     * @param string $paymentIntentId
     * @param array $paymentMethod
     * @return array
     */
    public function confirmPaymentWithMethod($paymentIntentId, $paymentMethod)
    {
        try {
            Log::info('Starting payment confirmation', [
                'payment_intent_id' => $paymentIntentId,
                'payment_method_type' => $paymentMethod['type'] ?? 'unknown'
            ]);

            // First retrieve the payment intent
            $paymentIntent = PaymentIntent::retrieve($paymentIntentId);
            
            Log::info('Retrieved payment intent for confirmation', [
                'payment_intent_id' => $paymentIntentId,
                'status' => $paymentIntent->status,
                'amount' => $paymentIntent->amount,
                'currency' => $paymentIntent->currency
            ]);

            // Prepare the confirmation parameters
            $confirmParams = [
                'payment_method' => [
                    'type' => $paymentMethod['type'],
                    'card' => [
                        'number' => $paymentMethod['card']['number'],
                        'exp_month' => $paymentMethod['card']['exp_month'],
                        'exp_year' => $paymentMethod['card']['exp_year'],
                        'cvc' => $paymentMethod['card']['cvc']
                    ],
                    'billing_details' => [
                        'name' => $paymentMethod['billing_details']['name'],
                        'email' => $paymentMethod['billing_details']['email'],
                        'phone' => $paymentMethod['billing_details']['phone'],
                        'address' => [
                            'line1' => $paymentMethod['billing_details']['address']['line1'],
                            'line2' => $paymentMethod['billing_details']['address']['line2'],
                            'city' => $paymentMethod['billing_details']['address']['city'],
                            'state' => $paymentMethod['billing_details']['address']['state'],
                            'postal_code' => $paymentMethod['billing_details']['address']['postal_code'],
                            'country' => $paymentMethod['billing_details']['address']['country']
                        ]
                    ]
                ]
            ];

            Log::info('Confirming payment intent with parameters', [
                'payment_intent_id' => $paymentIntentId,
                'confirm_params' => array_merge(
                    $confirmParams,
                    ['payment_method' => array_merge(
                        $confirmParams['payment_method'],
                        ['card' => ['number' => '****' . substr($paymentMethod['card']['number'], -4)]]
                    )]
                )
            ]);
            
            // Confirm the payment intent
            $confirmedIntent = $paymentIntent->confirm($confirmParams);

            Log::info('Payment intent confirmed', [
                'payment_intent_id' => $paymentIntentId,
                'status' => $confirmedIntent->status,
                'amount' => $confirmedIntent->amount,
                'currency' => $confirmedIntent->currency,
                'charges' => $confirmedIntent->charges ? [
                    'total_count' => $confirmedIntent->charges->total_count,
                    'data' => array_map(function($charge) {
                        return [
                            'id' => $charge->id,
                            'status' => $charge->status,
                            'amount' => $charge->amount,
                            'currency' => $charge->currency
                        ];
                    }, $confirmedIntent->charges->data)
                ] : null
            ]);

            return [
                'success' => true,
                'payment_intent' => $confirmedIntent,
                'status' => $confirmedIntent->status,
                'amount' => round($confirmedIntent->amount / 100, 2),
                'currency' => $confirmedIntent->currency
            ];
        } catch (ApiErrorException $e) {
            Log::error('Stripe API error during payment confirmation', [
                'error' => $e->getMessage(),
                'error_type' => get_class($e),
                'error_code' => $e->getStripeCode(),
                'http_status' => $e->getHttpStatus(),
                'payment_intent_id' => $paymentIntentId,
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
                'error_code' => $e->getStripeCode(),
                'http_status' => $e->getHttpStatus()
            ];
        } catch (\Exception $e) {
            Log::error('Unexpected error during payment confirmation', [
                'error' => $e->getMessage(),
                'error_type' => get_class($e),
                'payment_intent_id' => $paymentIntentId,
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
} 