<?php

use App\Http\Controllers\AdminUserController;
use App\Http\Controllers\AuthController;
use App\Http\Controllers\BrandController;
use App\Http\Controllers\CategoryController;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\ProductPricingController;
use App\Http\Controllers\VendorController;
use App\Http\Controllers\CustomerProfileController;
use App\Http\Controllers\CustomerAddressController;
use App\Http\Controllers\WishlistController;
use App\Http\Controllers\CartController;
use App\Http\Controllers\CheckoutController;
use App\Http\Controllers\OrderController;
use App\Http\Controllers\AdminOrderController;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application.
| Routes are loaded by the RouteServiceProvider and assigned to the "api" middleware group.
|
*/

// API version prefix
Route::prefix('v1')->group(function () {
    /*
    |--------------------------------------------------------------------------
    | Public Routes
    |--------------------------------------------------------------------------
    */

    // Authentication Routes
    Route::prefix('auth')->group(function () {
        Route::post('/register', [AuthController::class, 'register']);
        Route::post('/login', [AuthController::class, 'login']);
        Route::get('/redirect/{provider}', [AuthController::class, 'redirectToProvider']);
        Route::get('/callback/{provider}', [AuthController::class, 'handleProviderCallback']);
    });

    // Public Product Routes
    Route::prefix('products')->group(function () {
        Route::get('/', [ProductController::class, 'index']);
        Route::get('/homepage-sections', [ProductController::class, 'homepageSections']);
        Route::get('/{id}', [ProductController::class, 'show'])->where('id', '[0-9]+');
    });

    // Public Category Routes
    Route::prefix('categories')->group(function () {
        Route::get('/', [CategoryController::class, 'index']);
        Route::get('/tree', [CategoryController::class, 'tree']);
        Route::get('/featured', [CategoryController::class, 'featured']);
        Route::get('/{id}', [CategoryController::class, 'show'])->where('id', '[0-9]+');
        Route::get('/{id}/products', [CategoryController::class, 'products'])->where('id', '[0-9]+');
    });

    // Public Brand Routes
    Route::prefix('brands')->group(function () {
        Route::get('/', [BrandController::class, 'index']);
        Route::get('/{id}', [BrandController::class, 'show'])->where('id', '[0-9]+');
    });

    /*
    |--------------------------------------------------------------------------
    | Protected Routes (Authentication Required)
    |--------------------------------------------------------------------------
    */

    Route::middleware('auth:sanctum')->group(function () {
        // User Authentication Management
        Route::prefix('auth')->group(function () {
            Route::post('/logout', [AuthController::class, 'logout']);
            Route::get('/me', [AuthController::class, 'me']);
        });

        /*
        |--------------------------------------------------------------------------
        | Vendor Routes
        |--------------------------------------------------------------------------
        */
        Route::middleware('role:vendor')->prefix('vendor')->group(function () {
            // Vendor Product Management
            Route::prefix('products')->group(function () {
                Route::get('/', [ProductController::class, 'vendorProducts']);
                Route::post('/', [ProductController::class, 'store']);
                Route::get('/{id}', [ProductController::class, 'show'])->where('id', '[0-9]+');
                Route::put('/{id}', [ProductController::class, 'update'])->where('id', '[0-9]+');
                Route::patch('/{id}', [ProductController::class, 'update'])->where('id', '[0-9]+');
                Route::delete('/{id}', [ProductController::class, 'destroy'])->where('id', '[0-9]+');
                
                // Product Pricing Management
                Route::get('/{productId}/pricing', [ProductPricingController::class, 'getPricing'])->where('productId', '[0-9]+');
                Route::put('/{productId}/pack-price', [ProductPricingController::class, 'updatePackPrice'])->where('productId', '[0-9]+');
                Route::put('/{productId}/default-margins', [ProductPricingController::class, 'updateDefaultMargins'])->where('productId', '[0-9]+');
                
                // Bulk Pricing Management
                Route::post('/{productId}/bulk-prices', [ProductPricingController::class, 'createBulkPrice'])->where('productId', '[0-9]+');
                Route::put('/{productId}/bulk-prices', [ProductPricingController::class, 'updateAllBulkPrices'])->where('productId', '[0-9]+');
                Route::put('/{productId}/bulk-prices/order', [ProductPricingController::class, 'updateBulkPriceOrder'])->where('productId', '[0-9]+');
                Route::put('/bulk-prices/{id}', [ProductPricingController::class, 'updateBulkPrice'])->where('id', '[0-9]+');
                Route::delete('/bulk-prices/{id}', [ProductPricingController::class, 'deleteBulkPrice'])->where('id', '[0-9]+');
            });
        });

        /*
        |--------------------------------------------------------------------------
        | Buyer Routes
        |--------------------------------------------------------------------------
        */
        Route::middleware('role:buyer')->prefix('buyer')->group(function () {
            // Buyer specific routes
            // Add routes for cart, orders, favorites, etc.
        });

        /*
        |--------------------------------------------------------------------------
        | Admin Routes (SuperAdmin, Admin, Supervisor)
        |--------------------------------------------------------------------------
        */
        Route::middleware('role:superadmin,admin,supervisor')->prefix('admin')->group(function () {
            // Product Management
            Route::prefix('products')->group(function () {
                Route::get('/', [ProductController::class, 'index']);
                Route::post('/', [ProductController::class, 'store']);
                Route::get('/{id}', [ProductController::class, 'show'])->where('id', '[0-9]+');
                Route::put('/{id}', [ProductController::class, 'update'])->where('id', '[0-9]+');
                Route::patch('/{id}', [ProductController::class, 'update'])->where('id', '[0-9]+');
                Route::delete('/{id}', [ProductController::class, 'destroy'])->where('id', '[0-9]+');
                Route::patch('/{id}/toggle-approval', [ProductController::class, 'toggleApproval'])->where('id', '[0-9]+');
                Route::post('/bulk-action', [ProductController::class, 'bulkAction']);
            });

            // Brand Management
            Route::prefix('brands')->group(function () {
                Route::get('/', [BrandController::class, 'index']);
                Route::post('/', [BrandController::class, 'store']);
                Route::get('/{id}', [BrandController::class, 'show'])->where('id', '[0-9]+');
                Route::put('/{id}', [BrandController::class, 'update'])->where('id', '[0-9]+');
                Route::patch('/{id}', [BrandController::class, 'update'])->where('id', '[0-9]+');
                Route::delete('/{id}', [BrandController::class, 'destroy'])->where('id', '[0-9]+');
            });
        });

        /*
        |--------------------------------------------------------------------------
        | SuperAdmin & Admin Routes
        |--------------------------------------------------------------------------
        */
        Route::middleware('role:superadmin,admin')->prefix('admin')->group(function () {
            // User Management
            Route::prefix('users')->group(function () {
                Route::get('/', [AdminUserController::class, 'index']);
                Route::post('/', [AdminUserController::class, 'store']);
                Route::get('/{id}', [AdminUserController::class, 'show'])->where('id', '[0-9]+');
                Route::put('/{id}', [AdminUserController::class, 'update'])->where('id', '[0-9]+');
                Route::patch('/{id}', [AdminUserController::class, 'update'])->where('id', '[0-9]+');
                Route::delete('/{id}', [AdminUserController::class, 'destroy'])->where('id', '[0-9]+');
            });

            // Vendor Management
            Route::prefix('vendors')->group(function () {
                Route::get('/', [VendorController::class, 'index']);
                Route::patch('/{id}/approve', [VendorController::class, 'approve'])->where('id', '[0-9]+');
                Route::patch('/{id}/reject', [VendorController::class, 'reject'])->where('id', '[0-9]+');
            });
        });

        /*
        |--------------------------------------------------------------------------
        | Multiple Role Routes (Available to Multiple User Types)
        |--------------------------------------------------------------------------
        */
        Route::middleware('role:superadmin,admin,supervisor,vendor')->group(function () {
            // Category Management
            Route::prefix('categories')->group(function () {
                Route::post('/', [CategoryController::class, 'store']);
                Route::put('/{id}', [CategoryController::class, 'update'])->where('id', '[0-9]+');
                Route::patch('/{id}', [CategoryController::class, 'update'])->where('id', '[0-9]+');
                Route::delete('/{id}', [CategoryController::class, 'destroy'])->where('id', '[0-9]+');
                Route::patch('/{id}/status', [CategoryController::class, 'updateStatus'])->where('id', '[0-9]+');
                Route::post('/bulk-action', [CategoryController::class, 'bulkAction']);
            });
        });

        /*
        |--------------------------------------------------------------------------
        | SuperAdmin Only Routes
        |--------------------------------------------------------------------------
        */
        Route::middleware('role:superadmin')->prefix('superadmin')->group(function () {
            // System Settings
            // Route::get('/settings', [SettingController::class, 'index']);
            // Route::put('/settings', [SettingController::class, 'update']);
            
            // Currency Management
            // Route::apiResource('currencies', CurrencyController::class);
        });
    });

    // Customer Routes - Public
Route::prefix('auth')->group(function () {
    Route::post('/register', [AuthController::class, 'register']);
    Route::post('/login', [AuthController::class, 'login']);
});

// Customer Routes - Protected
Route::middleware('auth:sanctum')->group(function () {
    // Customer Profile Management
    Route::prefix('customer')->group(function () {
        // Profile
        Route::get('/profile', [CustomerProfileController::class, 'show']);
        Route::post('/profile', [CustomerProfileController::class, 'store']);
        
        // Addresses
        Route::get('/addresses', [CustomerAddressController::class, 'index']);
        Route::post('/addresses', [CustomerAddressController::class, 'store']);
        Route::get('/addresses/{id}', [CustomerAddressController::class, 'show']);
        Route::put('/addresses/{id}', [CustomerAddressController::class, 'update']);
        Route::delete('/addresses/{id}', [CustomerAddressController::class, 'destroy']);
        Route::patch('/addresses/{id}/set-default', [CustomerAddressController::class, 'setDefault']);
        
        // Wishlists
        Route::get('/wishlists', [WishlistController::class, 'index']);
        Route::post('/wishlists', [WishlistController::class, 'store']);
        Route::get('/wishlists/{id}', [WishlistController::class, 'show']);
        Route::put('/wishlists/{id}', [WishlistController::class, 'update']);
        Route::delete('/wishlists/{id}', [WishlistController::class, 'destroy']);
        Route::post('/wishlists/{id}/products', [WishlistController::class, 'addProduct']);
        Route::delete('/wishlists/{wishlistId}/products/{productId}', [WishlistController::class, 'removeProduct']);
    });
});

// Admin Customer Management Routes
Route::middleware('role:superadmin,admin,supervisor')->prefix('admin')->group(function () {
    Route::get('/customers', [CustomerProfileController::class, 'index']);
    Route::get('/customers/{id}', [CustomerProfileController::class, 'adminView']);
    Route::patch('/customers/{id}/verify', [CustomerProfileController::class, 'verify'])->middleware('role:superadmin,admin');
});

Route::middleware(['auth:sanctum', 'role:superadmin'])->prefix('superadmin')->group(function () {
    Route::get('/products', [ProductController::class, 'superadminProducts']);
});

/*
|--------------------------------------------------------------------------
| Cart Routes (Authenticated)
|--------------------------------------------------------------------------
*/
Route::middleware('auth:sanctum')->prefix('cart')->group(function () {
    Route::get('/', [CartController::class, 'getCart']);
    Route::post('/items', [CartController::class, 'addItem']);
    Route::put('/items/{id}', [CartController::class, 'updateItem']);
    Route::delete('/items/{id}', [CartController::class, 'removeItem']);
    Route::delete('/clear', [CartController::class, 'clearCart']);
    Route::put('/notes', [CartController::class, 'updateNotes']);
    Route::put('/shipping', [CartController::class, 'updateShipping']);
});

/*
|--------------------------------------------------------------------------
| Checkout Routes (Authenticated)
|--------------------------------------------------------------------------
*/
Route::middleware('auth:sanctum')->prefix('checkout')->group(function () {
    Route::post('/process', [CheckoutController::class, 'process']);
});

/*
|--------------------------------------------------------------------------
| Order Routes (Authenticated)
|--------------------------------------------------------------------------
*/
Route::middleware('auth:sanctum')->prefix('orders')->group(function () {
    Route::get('/', [OrderController::class, 'index']);
    Route::get('/{id}', [OrderController::class, 'show']);
});

/*
|--------------------------------------------------------------------------
| Admin Order Management
|--------------------------------------------------------------------------
*/
Route::middleware(['auth:sanctum', 'role:superadmin,admin,supervisor'])->prefix('admin/orders')->group(function () {
    Route::get('/', [AdminOrderController::class, 'index']);
    Route::get('/{id}', [AdminOrderController::class, 'show']);
    Route::patch('/{id}/status', [AdminOrderController::class, 'updateStatus']);
    Route::patch('/{id}/payment-status', [AdminOrderController::class, 'updatePaymentStatus']);
    Route::patch('/{id}/shipping-status', [AdminOrderController::class, 'updateShippingStatus']);
});

});

// Fallback for undefined routes
Route::fallback(function () {
    return response()->json([
        'success' => false,
        'message' => 'Not Found',
        'details' => 'The requested endpoint does not exist'
    ], 404);
});